import { useState } from 'react';
import { useMutation } from '@tanstack/react-query';
import { Button } from '@/components/ui/button';
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogHeader,
  DialogTitle,
  DialogTrigger,
} from '@/components/ui/dialog';
import {
  Form,
  FormControl,
  FormDescription,
  FormField,
  FormItem,
  FormLabel,
  FormMessage,
} from '@/components/ui/form';
import { Input } from '@/components/ui/input';
import { Textarea } from '@/components/ui/textarea';
import { RadioGroup, RadioGroupItem } from '@/components/ui/radio-group';
import { Checkbox } from '@/components/ui/checkbox';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { Separator } from '@/components/ui/separator';
import { useForm } from 'react-hook-form';
import { zodResolver } from '@hookform/resolvers/zod';
import { z } from 'zod';
import { 
  Download, 
  FileCode, 
  FileText, 
  Image, 
  Package, 
  Loader2, 
  Check,
  Globe,
  Smartphone,
  Monitor,
  Tablet
} from 'lucide-react';
import { toast } from '@/hooks/use-toast';
import { TemplateRenderer } from '@/services/template-renderer';
import type { WebsiteTemplate, TemplateCustomization } from '@shared/schema';

interface ExportDialogProps {
  template: WebsiteTemplate;
  customization: TemplateCustomization;
  children?: React.ReactNode;
  isOpen?: boolean;
  onOpenChange?: (open: boolean) => void;
}

type ExportFormat = 'html' | 'json' | 'image' | 'bundle';
type DeviceMode = 'desktop' | 'tablet' | 'mobile';

const exportSchema = z.object({
  format: z.enum(['html', 'json', 'image', 'bundle']),
  fileName: z.string()
    .min(1, 'File name is required')
    .max(100, 'File name must not exceed 100 characters')
    .regex(/^[a-zA-Z0-9_-]+$/, 'File name can only contain letters, numbers, hyphens, and underscores'),
  includeMetadata: z.boolean().default(true),
  deviceMode: z.enum(['desktop', 'tablet', 'mobile']).default('desktop'),
  optimizeForProduction: z.boolean().default(true),
  includeInteractivity: z.boolean().default(false),
});

type ExportFormData = z.infer<typeof exportSchema>;

export default function ExportDialog({
  template,
  customization,
  children,
  isOpen,
  onOpenChange
}: ExportDialogProps) {
  const [open, setOpen] = useState(false);
  const [exportProgress, setExportProgress] = useState<string>('');
  
  const isControlled = isOpen !== undefined && onOpenChange !== undefined;
  const dialogOpen = isControlled ? isOpen : open;
  const setDialogOpen = isControlled ? onOpenChange : setOpen;

  const form = useForm<ExportFormData>({
    resolver: zodResolver(exportSchema),
    defaultValues: {
      format: 'html',
      fileName: (template.name || 'template').toLowerCase().replace(/[^a-z0-9]/g, '_'),
      includeMetadata: true,
      deviceMode: 'desktop',
      optimizeForProduction: true,
      includeInteractivity: false,
    },
  });

  const watchedFormat = form.watch('format');

  // Export mutation
  const exportMutation = useMutation({
    mutationFn: async (data: ExportFormData) => {
      const templateRenderer = TemplateRenderer.create(template, customization);
      
      switch (data.format) {
        case 'html':
          return exportAsHTML(templateRenderer, data);
        case 'json':
          return exportAsJSON(data);
        case 'image':
          return exportAsImage(templateRenderer, data);
        case 'bundle':
          return exportAsBundle(templateRenderer, data);
        default:
          throw new Error('Invalid export format');
      }
    },
    onSuccess: (result) => {
      toast({
        title: "Export Complete!",
        description: result.message,
        duration: 4000,
      });
      setDialogOpen(false);
      setExportProgress('');
    },
    onError: (error: any) => {
      console.error('Export failed:', error);
      toast({
        title: "Export Failed",
        description: error.message || "Failed to export template. Please try again.",
        variant: "destructive",
      });
      setExportProgress('');
    },
  });

  const exportAsHTML = async (renderer: any, data: ExportFormData) => {
    setExportProgress('Generating HTML...');
    
    const html = renderer.generateHTML({
      deviceMode: data.deviceMode,
      includeInteractivity: data.includeInteractivity,
      optimizeForExport: data.optimizeForProduction,
    });

    // Add metadata comment if requested
    const metadataComment = data.includeMetadata ? `
<!-- 
  Template: ${template.name || 'Untitled'}
  Generated: ${new Date().toISOString()}
  Device Mode: ${data.deviceMode}
  IBrandBiz Template Builder
-->
` : '';

    const finalHTML = metadataComment + html;
    
    setExportProgress('Creating download...');
    
    const blob = new Blob([finalHTML], { type: 'text/html' });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = `${data.fileName}.html`;
    document.body.appendChild(a);
    a.click();
    document.body.removeChild(a);
    URL.revokeObjectURL(url);

    return { message: `HTML template exported as ${data.fileName}.html` };
  };

  const exportAsJSON = async (data: ExportFormData) => {
    setExportProgress('Preparing JSON data...');
    
    const exportData = {
      template: {
        id: template.id,
        name: template.name,
        description: template.description,
        type: template.templateType,
      },
      customization: customization,
      metadata: data.includeMetadata ? {
        exportedAt: new Date().toISOString(),
        deviceMode: data.deviceMode,
        version: '1.0.0',
        generator: 'IBrandBiz Template Builder',
      } : undefined,
    };

    setExportProgress('Creating download...');
    
    const blob = new Blob([JSON.stringify(exportData, null, 2)], { type: 'application/json' });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = `${data.fileName}.json`;
    document.body.appendChild(a);
    a.click();
    document.body.removeChild(a);
    URL.revokeObjectURL(url);

    return { message: `Template configuration exported as ${data.fileName}.json` };
  };

  const exportAsImage = async (renderer: any, data: ExportFormData) => {
    setExportProgress('Generating preview...');
    
    // Create a temporary container for rendering
    const container = document.createElement('div');
    container.style.position = 'absolute';
    container.style.left = '-9999px';
    container.style.top = '0';
    
    // Set dimensions based on device mode
    const dimensions = {
      desktop: { width: 1200, height: 800 },
      tablet: { width: 768, height: 1024 },
      mobile: { width: 375, height: 667 },
    };
    
    const { width, height } = dimensions[data.deviceMode];
    container.style.width = `${width}px`;
    container.style.height = `${height}px`;
    
    document.body.appendChild(container);
    
    try {
      setExportProgress('Rendering template...');
      
      // Generate and inject HTML
      const html = renderer.generateHTML({
        deviceMode: data.deviceMode,
        includeInteractivity: false,
        optimizeForExport: true,
      });
      
      container.innerHTML = html;
      
      setExportProgress('Capturing screenshot...');
      
      // Use html2canvas if available, otherwise create a simple canvas
      const canvas = document.createElement('canvas');
      canvas.width = width;
      canvas.height = height;
      const ctx = canvas.getContext('2d');
      
      if (ctx) {
        // Create a simple preview with template colors
        const colors = customization.colors;
        
        // Background
        ctx.fillStyle = colors.background;
        ctx.fillRect(0, 0, width, height);
        
        // Header
        ctx.fillStyle = colors.primary;
        ctx.fillRect(0, 0, width, height * 0.2);
        
        // Content areas
        ctx.fillStyle = colors.secondary;
        ctx.fillRect(width * 0.1, height * 0.3, width * 0.8, height * 0.15);
        
        ctx.fillStyle = colors.accent;
        ctx.fillRect(width * 0.1, height * 0.5, width * 0.25, height * 0.3);
        ctx.fillRect(width * 0.4, height * 0.5, width * 0.25, height * 0.3);
        ctx.fillRect(width * 0.7, height * 0.5, width * 0.25, height * 0.3);
        
        // Add text
        ctx.fillStyle = colors.text;
        ctx.font = '24px Inter, sans-serif';
        ctx.textAlign = 'center';
        ctx.fillText(template.name || 'Template Preview', width / 2, height * 0.1);
        
        setExportProgress('Creating download...');
        
        // Convert to blob and download
        canvas.toBlob((blob) => {
          if (blob) {
            const url = URL.createObjectURL(blob);
            const a = document.createElement('a');
            a.href = url;
            a.download = `${data.fileName}_${data.deviceMode}.png`;
            document.body.appendChild(a);
            a.click();
            document.body.removeChild(a);
            URL.revokeObjectURL(url);
          }
        }, 'image/png');
      }
      
    } finally {
      document.body.removeChild(container);
    }

    return { message: `Template preview exported as ${data.fileName}_${data.deviceMode}.png` };
  };

  const exportAsBundle = async (renderer: any, data: ExportFormData) => {
    setExportProgress('Creating bundle...');
    
    // This would require a zip library like JSZip
    // For now, we'll export multiple files sequentially
    const files = [];
    
    // HTML file
    const html = renderer.generateHTML({
      deviceMode: 'desktop',
      includeInteractivity: data.includeInteractivity,
      optimizeForExport: data.optimizeForProduction,
    });
    files.push({ name: `${data.fileName}.html`, content: html, type: 'text/html' });
    
    // JSON file
    const exportData = {
      template: {
        id: template.id,
        name: template.name,
        description: template.description,
        type: template.templateType,
      },
      customization: customization,
      metadata: {
        exportedAt: new Date().toISOString(),
        version: '1.0.0',
        generator: 'IBrandBiz Template Builder',
      },
    };
    files.push({ 
      name: `${data.fileName}_config.json`, 
      content: JSON.stringify(exportData, null, 2), 
      type: 'application/json' 
    });
    
    setExportProgress('Creating downloads...');
    
    // Download each file
    for (const file of files) {
      const blob = new Blob([file.content], { type: file.type });
      const url = URL.createObjectURL(blob);
      const a = document.createElement('a');
      a.href = url;
      a.download = file.name;
      document.body.appendChild(a);
      a.click();
      document.body.removeChild(a);
      URL.revokeObjectURL(url);
      
      // Small delay between downloads
      await new Promise(resolve => setTimeout(resolve, 500));
    }

    return { message: `Template bundle exported with ${files.length} files` };
  };

  const onSubmit = (data: ExportFormData) => {
    exportMutation.mutate(data);
  };

  const getFormatDescription = (format: ExportFormat) => {
    switch (format) {
      case 'html':
        return 'Complete standalone website that can be hosted anywhere';
      case 'json':
        return 'Template configuration file for backup or sharing';
      case 'image':
        return 'Preview screenshot of the template design';
      case 'bundle':
        return 'Complete package with HTML, JSON, and assets';
      default:
        return '';
    }
  };

  const getFormatIcon = (format: ExportFormat) => {
    switch (format) {
      case 'html': return FileCode;
      case 'json': return FileText;
      case 'image': return Image;
      case 'bundle': return Package;
      default: return Download;
    }
  };

  return (
    <Dialog open={dialogOpen} onOpenChange={setDialogOpen}>
      {children && (
        <DialogTrigger asChild>
          {children}
        </DialogTrigger>
      )}
      <DialogContent className="sm:max-w-[600px]" data-testid="dialog-export-template">
        <DialogHeader>
          <DialogTitle className="flex items-center gap-2">
            <Download className="w-5 h-5" />
            Export Template
          </DialogTitle>
          <DialogDescription>
            Export your template in various formats for different use cases.
          </DialogDescription>
        </DialogHeader>

        <Form {...form}>
          <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-6">
            <FormField
              control={form.control}
              name="format"
              render={({ field }) => (
                <FormItem>
                  <FormLabel>Export Format</FormLabel>
                  <FormControl>
                    <RadioGroup
                      onValueChange={field.onChange}
                      defaultValue={field.value}
                      className="space-y-3"
                    >
                      {(['html', 'json', 'image', 'bundle'] as ExportFormat[]).map((format) => {
                        const Icon = getFormatIcon(format);
                        return (
                          <Card key={format} className={`cursor-pointer transition-colors ${
                            field.value === format ? 'ring-2 ring-primary' : ''
                          }`}>
                            <CardHeader className="pb-2">
                              <div className="flex items-center space-x-3">
                                <RadioGroupItem value={format} id={format} />
                                <Icon className="w-5 h-5" />
                                <div className="flex-1">
                                  <CardTitle className="text-sm capitalize">{format.toUpperCase()}</CardTitle>
                                  <CardDescription className="text-xs">
                                    {getFormatDescription(format)}
                                  </CardDescription>
                                </div>
                                {format === 'bundle' && (
                                  <Badge variant="secondary" className="text-xs">Recommended</Badge>
                                )}
                              </div>
                            </CardHeader>
                          </Card>
                        );
                      })}
                    </RadioGroup>
                  </FormControl>
                  <FormMessage />
                </FormItem>
              )}
            />

            <FormField
              control={form.control}
              name="fileName"
              render={({ field }) => (
                <FormItem>
                  <FormLabel>File Name</FormLabel>
                  <FormControl>
                    <Input
                      placeholder="template_name"
                      {...field}
                      data-testid="input-export-filename"
                    />
                  </FormControl>
                  <FormDescription>
                    File name without extension (letters, numbers, hyphens, and underscores only).
                  </FormDescription>
                  <FormMessage />
                </FormItem>
              )}
            />

            {(watchedFormat === 'html' || watchedFormat === 'image') && (
              <FormField
                control={form.control}
                name="deviceMode"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Device Mode</FormLabel>
                    <FormControl>
                      <RadioGroup
                        onValueChange={field.onChange}
                        defaultValue={field.value}
                        className="flex space-x-6"
                      >
                        <div className="flex items-center space-x-2">
                          <RadioGroupItem value="desktop" id="desktop" />
                          <Monitor className="w-4 h-4" />
                          <label htmlFor="desktop" className="text-sm">Desktop</label>
                        </div>
                        <div className="flex items-center space-x-2">
                          <RadioGroupItem value="tablet" id="tablet" />
                          <Tablet className="w-4 h-4" />
                          <label htmlFor="tablet" className="text-sm">Tablet</label>
                        </div>
                        <div className="flex items-center space-x-2">
                          <RadioGroupItem value="mobile" id="mobile" />
                          <Smartphone className="w-4 h-4" />
                          <label htmlFor="mobile" className="text-sm">Mobile</label>
                        </div>
                      </RadioGroup>
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />
            )}

            <div className="space-y-4">
              <FormField
                control={form.control}
                name="includeMetadata"
                render={({ field }) => (
                  <FormItem className="flex flex-row items-start space-x-3 space-y-0">
                    <FormControl>
                      <Checkbox
                        checked={field.value}
                        onCheckedChange={field.onChange}
                        data-testid="checkbox-include-metadata"
                      />
                    </FormControl>
                    <div className="space-y-1 leading-none">
                      <FormLabel>Include Metadata</FormLabel>
                      <FormDescription>
                        Add creation date, template info, and generator details.
                      </FormDescription>
                    </div>
                  </FormItem>
                )}
              />

              {watchedFormat === 'html' && (
                <>
                  <FormField
                    control={form.control}
                    name="optimizeForProduction"
                    render={({ field }) => (
                      <FormItem className="flex flex-row items-start space-x-3 space-y-0">
                        <FormControl>
                          <Checkbox
                            checked={field.value}
                            onCheckedChange={field.onChange}
                            data-testid="checkbox-optimize-production"
                          />
                        </FormControl>
                        <div className="space-y-1 leading-none">
                          <FormLabel>Optimize for Production</FormLabel>
                          <FormDescription>
                            Minify CSS and optimize for faster loading.
                          </FormDescription>
                        </div>
                      </FormItem>
                    )}
                  />

                  <FormField
                    control={form.control}
                    name="includeInteractivity"
                    render={({ field }) => (
                      <FormItem className="flex flex-row items-start space-x-3 space-y-0">
                        <FormControl>
                          <Checkbox
                            checked={field.value}
                            onCheckedChange={field.onChange}
                            data-testid="checkbox-include-interactivity"
                          />
                        </FormControl>
                        <div className="space-y-1 leading-none">
                          <FormLabel>Include JavaScript</FormLabel>
                          <FormDescription>
                            Add interactive features like forms and animations.
                          </FormDescription>
                        </div>
                      </FormItem>
                    )}
                  />
                </>
              )}
            </div>

            {exportProgress && (
              <div className="flex items-center justify-center p-4 bg-muted rounded-lg">
                <Loader2 className="w-4 h-4 mr-2 animate-spin" />
                <span className="text-sm">{exportProgress}</span>
              </div>
            )}

            <Separator />

            <div className="flex justify-end space-x-3">
              <Button
                type="button"
                variant="outline"
                onClick={() => setDialogOpen(false)}
                disabled={exportMutation.isPending}
                data-testid="button-cancel-export"
              >
                Cancel
              </Button>
              <Button
                type="submit"
                disabled={exportMutation.isPending}
                className="min-w-[120px]"
                data-testid="button-start-export"
              >
                {exportMutation.isPending ? (
                  <>
                    <Loader2 className="w-4 h-4 mr-2 animate-spin" />
                    Exporting...
                  </>
                ) : exportMutation.isSuccess ? (
                  <>
                    <Check className="w-4 h-4 mr-2" />
                    Exported
                  </>
                ) : (
                  <>
                    <Download className="w-4 h-4 mr-2" />
                    Export
                  </>
                )}
              </Button>
            </div>
          </form>
        </Form>
      </DialogContent>
    </Dialog>
  );
}