import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Label } from '@/components/ui/label';
import { Textarea } from '@/components/ui/textarea';
import { Input } from '@/components/ui/input';
import { Button } from '@/components/ui/button';
import { Plus, Trash2 } from 'lucide-react';
import type { WebsiteTemplate } from '@shared/schema';

interface ContentSection {
  id: string;
  type: string;
  content: any;
  order: number;
}

interface Content {
  sections: ContentSection[];
}

interface ContentEditorProps {
  content: Content;
  onChange: (content: Content) => void;
  template: WebsiteTemplate;
}

export default function ContentEditor({ content, onChange, template }: ContentEditorProps) {
  const handleSectionChange = (sectionId: string, field: string, value: any) => {
    const updatedSections = content.sections.map(section => 
      section.id === sectionId 
        ? { ...section, content: { ...section.content, [field]: value } }
        : section
    );
    
    onChange({
      ...content,
      sections: updatedSections
    });
  };

  const addSection = () => {
    const newSection: ContentSection = {
      id: `section-${Date.now()}`,
      type: 'text',
      content: { text: '', title: '' },
      order: content.sections.length
    };
    
    onChange({
      ...content,
      sections: [...content.sections, newSection]
    });
  };

  const removeSection = (sectionId: string) => {
    const updatedSections = content.sections.filter(section => section.id !== sectionId);
    onChange({
      ...content,
      sections: updatedSections
    });
  };

  return (
    <Card>
      <CardHeader>
        <CardTitle className="flex items-center justify-between">
          Content Editor
          <Button
            variant="outline"
            size="sm"
            onClick={addSection}
            data-testid="button-add-section"
          >
            <Plus className="w-4 h-4 mr-2" />
            Add Section
          </Button>
        </CardTitle>
      </CardHeader>
      <CardContent className="space-y-6">
        {content.sections.length === 0 ? (
          <div className="text-center py-8 text-muted-foreground">
            <p>No content sections yet.</p>
            <p className="text-sm">Click "Add Section" to get started.</p>
          </div>
        ) : (
          content.sections.map((section, index) => (
            <div key={section.id} className="border rounded-lg p-4 space-y-4">
              <div className="flex items-center justify-between">
                <Label className="font-medium">Section {index + 1}</Label>
                <Button
                  variant="ghost"
                  size="sm"
                  onClick={() => removeSection(section.id)}
                  data-testid={`button-remove-section-${section.id}`}
                >
                  <Trash2 className="w-4 h-4" />
                </Button>
              </div>
              
              <div className="space-y-3">
                <div className="space-y-2">
                  <Label htmlFor={`title-${section.id}`}>Title</Label>
                  <Input
                    id={`title-${section.id}`}
                    value={section.content.title || ''}
                    onChange={(e) => handleSectionChange(section.id, 'title', e.target.value)}
                    placeholder="Section title..."
                    data-testid={`input-section-title-${section.id}`}
                  />
                </div>
                
                <div className="space-y-2">
                  <Label htmlFor={`content-${section.id}`}>Content</Label>
                  <Textarea
                    id={`content-${section.id}`}
                    value={section.content.text || ''}
                    onChange={(e) => handleSectionChange(section.id, 'text', e.target.value)}
                    placeholder="Section content..."
                    rows={4}
                    data-testid={`textarea-section-content-${section.id}`}
                  />
                </div>
              </div>
            </div>
          ))
        )}
      </CardContent>
    </Card>
  );
}