import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Label } from '@/components/ui/label';
import { Input } from '@/components/ui/input';
import { Button } from '@/components/ui/button';
import { RefreshCw } from 'lucide-react';

interface Colors {
  primary: string;
  secondary: string;
  accent: string;
  background: string;
  text: string;
}

interface ColorCustomizerProps {
  colors: Colors;
  onChange: (colors: Colors) => void;
}

export default function ColorCustomizer({ colors, onChange }: ColorCustomizerProps) {
  const handleColorChange = (key: keyof Colors, value: string) => {
    onChange({
      ...colors,
      [key]: value,
    });
  };

  const resetColors = () => {
    onChange({
      primary: '#00C851',
      secondary: '#427AB9',
      accent: '#FF6F00',
      background: '#FFFFFF',
      text: '#2B2B2D'
    });
  };

  return (
    <Card>
      <CardHeader>
        <CardTitle className="flex items-center justify-between">
          Color Palette
          <Button
            variant="ghost"
            size="sm"
            onClick={resetColors}
            data-testid="button-reset-colors"
          >
            <RefreshCw className="w-4 h-4" />
          </Button>
        </CardTitle>
      </CardHeader>
      <CardContent className="space-y-4">
        {Object.entries(colors).map(([key, value]) => (
          <div key={key} className="space-y-2">
            <Label htmlFor={`color-${key}`} className="capitalize">
              {key}
            </Label>
            <div className="flex items-center space-x-2">
              <Input
                id={`color-${key}`}
                type="color"
                value={value}
                onChange={(e) => handleColorChange(key as keyof Colors, e.target.value)}
                className="w-12 h-8 p-1 rounded cursor-pointer"
                data-testid={`input-color-${key}`}
              />
              <Input
                type="text"
                value={value}
                onChange={(e) => handleColorChange(key as keyof Colors, e.target.value)}
                className="flex-1"
                data-testid={`input-color-hex-${key}`}
              />
            </div>
          </div>
        ))}
      </CardContent>
    </Card>
  );
}