import React, { useEffect, useMemo, useState } from "react";
import { loadStripe } from "@stripe/stripe-js";
import { Elements, PaymentElement, useElements, useStripe } from "@stripe/react-stripe-js";
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogDescription, DialogFooter } from "@/components/ui/dialog";
import { Button } from "@/components/ui/button";
import { toast } from "@/hooks/use-toast";
import { apiRequest } from "@/lib/queryClient";

// Use the new publishable key variable
const stripeKey = import.meta.env.VITE_STRIPE_PUBLISHABLE_KEY as string;
console.log('Stripe key check:', stripeKey ? `EXISTS (starts with: ${stripeKey.substring(0, 7)})` : 'MISSING');
const stripePromise = loadStripe(stripeKey);

type Props = {
  open: boolean;
  onOpenChange: (o: boolean) => void;
  priceId: string;
  planLabel?: string;
  userEmail?: string | null;
};

export default function SubscribeModal({ open, onOpenChange, priceId, planLabel = "", userEmail }: Props) {
  const [clientSecret, setClientSecret] = useState<string | null>(null);
  const [loading, setLoading] = useState(false);
  const [initError, setInitError] = useState<string | null>(null);

  // create subscription → get clientSecret
  useEffect(() => {
    if (!open) { setClientSecret(null); setInitError(null); return; }
    let cancelled = false;
    (async () => {
      try {
        setLoading(true);
        setInitError(null);
        const res = await apiRequest<{clientSecret: string; subscriptionId: string}>("POST", "/api/billing/create-subscription", { billingType: priceId, email: userEmail });
        if (!cancelled) setClientSecret(res.clientSecret); // must be a non-empty string
      } catch (e: any) {
        console.error("[create-subscription] error:", e);
        if (!cancelled) setInitError(e?.message || "Could not initialize Stripe.");
      } finally {
        if (!cancelled) setLoading(false);
      }
    })();
    return () => { cancelled = true; };
  }, [open, priceId, userEmail]);

  // IMPORTANT: Elements options must be memoized AND keyed by clientSecret
  const options = useMemo(() => clientSecret ? ({
    clientSecret,
    appearance: {
      theme: "stripe",
      variables: { borderRadius: "12px" }
    }
  }) : undefined, [clientSecret]);

  return (
    <Dialog open={open} onOpenChange={onOpenChange}>
      <DialogContent className="sm:max-w-lg" forceMount>
        <DialogHeader>
          <DialogTitle>Complete Your Pro Subscription</DialogTitle>
          {planLabel && <DialogDescription className="text-base font-medium">{planLabel}</DialogDescription>}
        </DialogHeader>

        {initError && (
          <div className="text-sm text-red-600 bg-red-50 rounded-md p-3 mb-2">
            {initError}
          </div>
        )}

        {initError && (
          <div className="text-sm text-red-600 bg-red-50 rounded-md p-3 mb-2">
            {initError}
          </div>
        )}

        {loading && (
          <div className="py-6 text-center text-sm text-muted-foreground">
            Initializing secure payment…
          </div>
        )}

        {!loading && clientSecret && (
          <Elements key={clientSecret} stripe={stripePromise} options={options}>
            <InnerForm onClose={() => onOpenChange(false)} />
          </Elements>
        )}

        {!loading && !clientSecret && !initError && (
          <div className="py-6 text-center text-sm text-muted-foreground">
            Waiting for secure payment initialization…
          </div>
        )}
      </DialogContent>
    </Dialog>
  );
}

function InnerForm({ onClose }: { onClose: () => void }) {
  const stripe = useStripe();
  const elements = useElements();
  const [submitting, setSubmitting] = useState(false);
  const [formError, setFormError] = useState<string | null>(null);

  const handleSubmit: React.FormEventHandler<HTMLFormElement> = async (e) => {
    e.preventDefault();
    setFormError(null);

    if (!stripe || !elements) {
      setFormError("Payment is still loading. Please wait a second…");
      return;
    }

    setSubmitting(true);
    const { error } = await stripe.confirmPayment({
      elements,
      confirmParams: { return_url: `${window.location.origin}/billing/return` },
      redirect: "if_required",
    });

    if (error) {
      console.error("[confirmPayment] ", error);
      setFormError(error.message || "Payment failed. Try a different method.");
      toast({ title: "Payment failed", description: error.message || "Try again.", variant: "destructive" });
      setSubmitting(false);
      return;
    }

    toast({ title: "Subscribed 🎉", description: "Your Pro subscription is active." });
    setSubmitting(false);
    onClose();
  };

  return (
    <form onSubmit={handleSubmit} className="space-y-4">
      {/* Keep a stable area so CSS/animation never hides the iframe */}
      <div style={{ minHeight: 190 }}>
        <PaymentElement />
      </div>

      {formError && <div className="text-sm text-red-600 bg-red-50 rounded-md p-2">{formError}</div>}

      <DialogFooter className="gap-2">
        <Button type="button" variant="outline" onClick={onClose} disabled={submitting}>
          Cancel
        </Button>
        <Button type="submit" disabled={!stripe || !elements || submitting}>
          {submitting ? "Processing…" : "Subscribe Monthly"}
        </Button>
      </DialogFooter>
    </form>
  );
}