/**
 * AI Button Component - Simplified drawer-style AI Assistant
 * Single "AI Assist" button that opens a drawer with optional prompt field
 */

import { useState } from "react";
import { Button } from "@/components/ui/button";
import { Textarea } from "@/components/ui/textarea";
import { Label } from "@/components/ui/label";
import { Card, CardContent } from "@/components/ui/card";
import { 
  Sparkles, 
  RefreshCw, 
  Plus, 
  Minus, 
  Loader2,
  ChevronDown,
  ChevronUp,
  MessageSquare
} from "lucide-react";
import { useToast } from "@/hooks/use-toast";
import { aiClient, AiClientError } from "@/services/ai/aiClient";
import type { AiAction, AiJobStatus } from "@shared/ai-types";
import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogDescription,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
} from "@/components/ui/alert-dialog";

interface AIButtonProps {
  sectionKind: string;
  sectionTitle: string;
  existingContent?: string;
  onContentUpdate: (content: string) => void;
  context?: {
    businessName?: string;
    industry?: string;
    description?: string;
    businessBrief?: any;
  };
  disabled?: boolean;
  className?: string;
}

export function AIButton({
  sectionKind,
  sectionTitle,
  existingContent = "",
  onContentUpdate,
  context,
  disabled = false,
  className = "",
}: AIButtonProps) {
  const [isDrawerOpen, setIsDrawerOpen] = useState(false);
  const [userPrompt, setUserPrompt] = useState("");
  const [status, setStatus] = useState<AiJobStatus>("idle");
  const [lastAction, setLastAction] = useState<AiAction | null>(null);
  const [showReplaceDialog, setShowReplaceDialog] = useState(false);
  const [pendingGenerate, setPendingGenerate] = useState(false);
  const [statusMessage, setStatusMessage] = useState("Generating...");
  const { toast } = useToast();

  const hasContent = existingContent.trim().length > 0;
  const hasBrief = context?.businessBrief && Object.values(context.businessBrief).some(v => v);
  const isLoading = status === "loading";

  // Execute AI action with error handling
  const executeAiAction = async (action: AiAction, append: boolean = false) => {
    if (disabled) return;

    setStatus("loading");
    setLastAction(action);
    setStatusMessage("Drafting…");

    // Show updating status after 3 seconds
    const statusTimer = setTimeout(() => {
      setStatusMessage("Still working…");
    }, 3000);

    try {
      let response;

      // Call appropriate AI client method based on action
      // Default to Professional tone and Standard length
      switch (action) {
        case "generate":
          response = await aiClient.generateContent(
            sectionKind,
            sectionTitle,
            "Professional", // Default tone
            context,
            existingContent,
            "Standard", // Default length
            userPrompt
          );
          break;
        case "rephrase":
          response = await aiClient.rephraseContent(
            sectionKind,
            sectionTitle,
            existingContent,
            "Professional",
            context,
            userPrompt
          );
          break;
        case "expand":
          response = await aiClient.expandContent(
            sectionKind,
            sectionTitle,
            existingContent,
            "Professional",
            context,
            userPrompt
          );
          break;
        case "summarize":
          response = await aiClient.summarizeContent(
            sectionKind,
            sectionTitle,
            existingContent,
            "Professional",
            context,
            userPrompt
          );
          break;
        default:
          throw new Error(`Unknown action: ${action}`);
      }

      // Handle successful response
      clearTimeout(statusTimer);
      if (response.success && response.content) {
        const newContent = append ? `${existingContent}\n\n${response.content}` : response.content;
        onContentUpdate(newContent);
        
        // Use SN's toast messages
        toast({
          description: "Draft generated",
        });
        
        setStatus("success");
        setStatusMessage("Generating...");
        // Clear user prompt after successful generation
        if (action === "generate") {
          setUserPrompt("");
          setIsDrawerOpen(false);
        }
      } else if (!response.success) {
        throw new Error(response.error || "AI request failed — please try again");
      }

    } catch (error) {
      clearTimeout(statusTimer);
      console.error("AI action failed:", error);
      
      let errorMessage = "AI request failed — please try again";

      if (error instanceof AiClientError) {
        errorMessage = error.message;
      } else if (error instanceof Error) {
        errorMessage = error.message;
      }

      toast({
        description: errorMessage,
        variant: "destructive",
      });

      setStatus("error");
      setStatusMessage("Generating...");
    }
  };

  // Handle generate button click
  const handleGenerate = () => {
    // Show toast if no business brief (SN's spec: "Add your Business Brief for better results (optional).")
    if (!hasBrief) {
      toast({
        description: "Add your Business Brief for better results (optional).",
        duration: 4000,
      });
    }

    // If existing content, show replace/append dialog
    if (hasContent) {
      setPendingGenerate(true);
      setShowReplaceDialog(true);
      return;
    }

    // Otherwise execute directly
    executeAiAction("generate", false);
  };

  // Handle dialog response
  const handleDialogResponse = (append: boolean) => {
    setShowReplaceDialog(false);
    if (pendingGenerate) {
      executeAiAction("generate", append);
      setPendingGenerate(false);
    }
  };

  return (
    <>
      {/* Main AI Assist Button */}
      <div className={className}>
        <Button
          variant="outline"
          onClick={() => setIsDrawerOpen(!isDrawerOpen)}
          disabled={disabled}
          className="w-full justify-between"
          data-testid="button-ai-assist"
        >
          <div className="flex items-center gap-2">
            <MessageSquare className="w-4 h-4" />
            <span>Prompt AI</span>
          </div>
          {isDrawerOpen ? <ChevronUp className="w-4 h-4" /> : <ChevronDown className="w-4 h-4" />}
        </Button>

        {/* Drawer Content */}
        {isDrawerOpen && (
          <Card className="mt-2">
            <CardContent className="pt-4 space-y-3">
              <div className="space-y-2">
                <Label className="text-sm font-medium">Optional Prompt</Label>
                <Textarea
                  value={userPrompt}
                  onChange={(e) => setUserPrompt(e.target.value)}
                  placeholder="Add any specifics for this section (bullets, tone, key points)…"
                  className="min-h-[80px] text-sm"
                  data-testid="textarea-user-prompt"
                />
              </div>

              <div className="space-y-2">
                <Button
                  onClick={handleGenerate}
                  disabled={isLoading}
                  className="w-full"
                  data-testid="button-generate-section"
                >
                  {isLoading && lastAction === "generate" ? (
                    <>
                      <Loader2 className="w-4 h-4 mr-2 animate-spin" />
                      Generating...
                    </>
                  ) : (
                    <>
                      <Sparkles className="w-4 h-4 mr-2" />
                      Generate Section
                    </>
                  )}
                </Button>
                <p className="text-xs text-muted-foreground text-center">
                  Will use your Business Brief + this prompt to create a draft.
                </p>
              </div>
            </CardContent>
          </Card>
        )}
      </div>

      {/* Replace or Append Dialog */}
      <AlertDialog open={showReplaceDialog} onOpenChange={setShowReplaceDialog}>
        <AlertDialogContent>
          <AlertDialogHeader>
            <AlertDialogTitle>Generate new content?</AlertDialogTitle>
            <AlertDialogDescription>
              This section already has text. Choose how to apply the new AI-generated content.
            </AlertDialogDescription>
          </AlertDialogHeader>
          <AlertDialogFooter>
            <AlertDialogCancel>
              Cancel
            </AlertDialogCancel>
            <Button
              variant="outline"
              onClick={() => handleDialogResponse(false)}
            >
              Replace Draft
            </Button>
            <AlertDialogAction onClick={() => handleDialogResponse(true)}>
              Append to Draft
            </AlertDialogAction>
          </AlertDialogFooter>
        </AlertDialogContent>
      </AlertDialog>
    </>
  );
}
