import { useState, useEffect } from "react";
import { useQuery } from "@tanstack/react-query";

interface VisitorLocation {
  latitude: number;
  longitude: number;
  country: string;
  countryCode?: string;
  city?: string;
  count: number;
  sessions: number;
}

interface VisitorLocationsResponse {
  locations: VisitorLocation[];
  countries: Array<{
    country: string;
    countryCode: string;
    visitors: number;
    pageViews: number;
  }>;
  summary: {
    totalLocations: number;
    totalCountries: number;
    totalSessions: number;
    dateRange: {
      days: number;
      description: string;
    };
  };
}

interface WorldMapProps {
  className?: string;
}

// Professional world map - we'll embed the SVG content directly for better performance and accuracy
const WORLD_MAP_VIEWBOX = "0 0 939 627";
const WORLD_MAP_SCALE = 1000 / 939; // Scale to fit our 1000x500 viewBox

// Convert lat/lng to SVG coordinates (simplified Mercator-like projection)
function latLngToSVG(lat: number, lng: number): { x: number; y: number } {
  const mapWidth = 1000;
  const mapHeight = 500;
  
  // Simple linear projection (good enough for dot positioning)
  const x = ((lng + 180) / 360) * mapWidth;
  const y = ((90 - lat) / 180) * mapHeight;
  
  return { x, y };
}

// Get dot size based on visitor count
function getDotSize(count: number, maxCount: number): number {
  const minSize = 3;
  const maxSize = 12;
  const ratio = count / maxCount;
  return minSize + (maxSize - minSize) * ratio;
}

// Get dot color based on visitor count (heat map style)
function getDotColor(count: number, maxCount: number): string {
  const ratio = count / maxCount;
  
  if (ratio > 0.8) return "#dc2626"; // Red for high activity
  if (ratio > 0.6) return "#ea580c"; // Orange-red
  if (ratio > 0.4) return "#f59e0b"; // Orange
  if (ratio > 0.2) return "#eab308"; // Yellow
  return "#22c55e"; // Green for low activity
}

export function WorldMap({ className = "" }: WorldMapProps) {
  const [selectedLocation, setSelectedLocation] = useState<VisitorLocation | null>(null);
  const [hoveredLocation, setHoveredLocation] = useState<VisitorLocation | null>(null);

  const { data: locationsData, isLoading, error } = useQuery<VisitorLocationsResponse>({
    queryKey: ['/api/admin/visitor-locations'],
    refetchInterval: 5 * 60 * 1000, // Refresh every 5 minutes
  });

  if (isLoading) {
    return (
      <div className={`flex items-center justify-center h-64 bg-gray-50 dark:bg-gray-800 rounded-lg ${className}`}>
        <div className="text-center">
          <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600 mx-auto mb-2" />
          <p className="text-sm text-gray-600 dark:text-gray-400">Loading world map...</p>
        </div>
      </div>
    );
  }

  if (error) {
    return (
      <div className={`flex items-center justify-center h-64 bg-gray-50 dark:bg-gray-800 rounded-lg ${className}`}>
        <div className="text-center">
          <p className="text-sm text-red-600 dark:text-red-400">Failed to load visitor locations</p>
          <p className="text-xs text-gray-500 dark:text-gray-500 mt-1">World map unavailable</p>
        </div>
      </div>
    );
  }

  const locations = locationsData?.locations || [];
  const maxCount = Math.max(...locations.map(loc => loc.count));

  return (
    <div className={`relative ${className}`} data-testid="world-map">
      <div className="bg-white dark:bg-gray-900 rounded-lg border border-gray-200 dark:border-gray-700 overflow-hidden">
        <svg
          viewBox="0 0 1000 500"
          className="w-full h-full"
          style={{ minHeight: '250px', maxHeight: '400px' }}
        >
          {/* World map background */}
          <defs>
            <pattern id="ocean" patternUnits="userSpaceOnUse" width="4" height="4">
              <rect width="4" height="4" fill="#f0f9ff" className="dark:fill-gray-800" />
            </pattern>
          </defs>
          
          {/* Ocean background */}
          <rect width="1000" height="500" fill="url(#ocean)" />
          
          {/* Professional world map continents - import from the detailed SVG */}
          <g transform={`scale(${WORLD_MAP_SCALE}, ${WORLD_MAP_SCALE * 0.8})`}>
            {/* Main continents and major landmasses */}
            <path d="M793.97,479.53c-3.21-4.42-.76-11.03-6.97-11.53.02-1.56.07-5.65-1.99-5.99.17,5.62.37,17.93-8.01,13.99-5.58-4.08-11.12-4.52-8-13-2.29.24-4.61-.73-6.88-.35-.32.05-5.49,3.04-5.7,3.28-.64.73,1.12,3.53.59,4.06-.29.28-7.69,1.25-7.99.98-.33-.3.8-4.08-1.51-2.97-2.02.97-5.12,7.58-5.65,7.88-1.27.72-3.55-1.19-3.85-.9-.19.18-.21,3.36-1.5,5.02-1.3,1.68-3.23,1.61-4.51,2.99-3.99-.07-6.78.92-10,3-1.62,1.04-3.18,2.18-4,4-1.6,3.56.54,3.54.96,6.56.41,2.96.5,3.69,1.04,6.44,1.01,5.17,1.85,8.53,2,14,.03,1-.04,2.01,0,3,1.57-.17,3.55.27,5.01,0,1.03-.19,2.34-1.53,3.7-1.77,2.84-.51,6.42.65,8.3-1.23,1.08-.29.89-1.39.95-1.84.01-.06.05-.1.05-.16,3.67-.89,7.06-2.3,11-3,7.31-1.3,15.95-2.07,18.01,7,2.71-1.57,2.98-5.92,6.98-5-.06,2.06-.11,4.09.44,6.11.52,1.91,2.16,1.82,2.56,2.6,2.14,4.11,1.2,7.56,7.27,8.65,2.86.51,6.71-4.17,8.23.64,1.01-2.92,6.52-3.46,7.28-4.22.43-.43,3.16-9.42,4.05-11.45,1.88-4.29,4.95-5.88,5.06-11.93.19-10.22-7.64-13.05-11.86-20.92-1.52-1.49-3.77-2.18-5.06-3.94Z" className="cls-1" />
            
            {/* North America */}
            <path d="M228,73c1.58.44,3.59,1.11,5,2-1.06,2.27-6.31,6.43-3,7,2.3.4,4.15-.94,5-3,3.87-1.15,2.52-.34,5,0-.66,3.71,2.29.91,5,1,.27.7-.28,2.67,0,3-1.75-.33-11.98,6.38-7.49,7,4.08.57,1.66-2.1,3.47-3.55.94-.76,2.36-.22,3.02-.46l3.49,2c4.34-2.32,2.53-8.77,7.75-11.33,5.19-2.55,3.51.34,2.76,3.34l-2.99,3.51c-.04.94,1.32.39,1.99.49.94.14,2.03-.12,3,0,1.31,6.17-2.73,2.72-4,5,.05.33-.06.68,0,1l-11,5.01c1.41,4.53,7.22,5.21,6,9.98-6.24,1.37-6.13-3.71-10-6.99-1.55-1.31-1.68-2.74-5-1.99l1,.99c-.65,2.75,5.52,13.32,7.47,13.6.84.12,3.14-.98,3.53-.59,1.33,1.32-2.11,9.32-5.99,7-.99-4.85-6.92-2.4-7.98,1.52-.62,2.29,2.77,1.67,2.98,1.48.66-.61-1.45-2.9,1.5-3,2.32-.08,1.58,1.6,2.5,3-.47,1.75-1.93,4.87-1.51,6,1.19,3.22,2.22-2.18,3.01-3,5.46-5.68,2.19,3.69,2.5,4,.28.28,2.22-.19,3,0-.46,3.07,5.27,4.05,5.99,1.41.16-.59-.99-1.27-.99-1.41.02-4.81,1.63-2.85,3-4,1.18-.98.68-1.23,1-2,1.26-3.02.39-4.69,4-6.99-4.78-1.68-3.57.25-4.25.61-4.08,2.17-3.39-4.97-2.23-7.6,1.95-4.42,6.8-2.06,10.47-3.03.39-1.49-1.34-.92-1.89-1.56-4.48-5.24,3.52-3.04,6.89-2.44.44-1.63-1.75-.51-2.02-1.62-.23-.95.16-2.29.01-3.37h4.01c-.01-.97.16-2.07,0-3,.24-1.2,1.61-.74,2-1,3.08-2.03,5.35-7.71,7.99-10.51-1.54-1.17-4.88,1.57-5.99,2.51-2.63.78-2.48-2.12-2.06-3.44.31-.99,3.42-.54,4.06-2.06-1.06-1-1.81.04-2.06-1.95-.44-3.56,1.79-2.27,2.06-2.54,2.24-.1,5.47,1.09,5-2,3-.44,3.27-.07,6-2,2.75-1.95,1.78-2.03,3-4,.32-.52,2.51-.86,2-2.48-5.15-2.7-5.68-.33-5-7.52-4.9-1.15-7.22,3.79-11.99,3-.93-3.98,2.67-3.65,3.99-5.98-3.04.21-5.49-.69-7.5-2.99-1.42,5.2-5.65,2.7-4.5,7.99-4.43,2.13-6.48-9.76-10-7-.38.3,4.65,7.27,2.99,8.98-2.76,2.85-10.73-13.33-7.99-2-2.61-.64-4.97,1.34-7.01,3-2.26,1.84.38,2.75,2,4,.55,1.05,1.41,1.96,2,3-.13.94.13,2.03,0,3-1.5-.86-8.52-1.83-8.87-2.12-.4-.32-.33-3.01-1.06-4.19-1.55-2.51-6.3-3.44-4.97.7.54,1.69,4.8,1.81,3.9,5.09-1.12,4.05-7.35-1.59-8.51-1.46-1.24.14-.88,6.31-6.48,3.98-.25,1.85,1.95.7,3,.99" className="cls-1" />
            
            {/* Greenland */}
            <path d="M184.99,173.51l1.01,9.49c-5.73,1.45-4.15-3.83-5.01-8-1.6-.25-.63,1.65-2.19,2.33-7.25,3.19-3.64-3.37-6.31-4.34-.33-.12-2.9,5.52-5.49,2.99-1.18-1.15,2.55-5.49-.49-4.98-5.01,1.88-4.8,5.8-7.51,9.49,2.56,1.93,6,1.51,9,1.5,.96,3.3-2.26,3.74-5.01,5-.41.19-1.38-.12-1.99.01-1.35.86,5.3,3.18,6.99.99,1.91.4,6.32.19,7.59,1.97,1.18,1.64-.52,3.1-.59,4.03-4.26-.94-7.74-.44-12,0-.28.2,1.36,2.5,2,2.99,2.25,1.73,4.53.38,6.08,1.43,2.07,1.4-1.86,6.87,6.97,4.12,4.09-1.27,6.66-6.22,10.95-5.54,1.47,1.36,2.43,3.38,4.99,2.99.13-.62-.17-1.53,0-1.99.3-.8.07-1.6.9-2.56,1.03-1.19,3.2-.87,3.09-2.43-9.21-1.88-6.25-11.04-9-17.51-.87-2.04-1.49-3.67-4.01-2ZM165.39,195.01s-.07.06-.1.09l-.28-.08h.39Z" className="cls-1" />
            
            {/* Asia - Major portions */}
            <path d="M563.56,405.04c-4.8.61-9.63,2.9-14.56,2.96-4.98-3.16-2.82-1.98-4.2-6.32-.43-1.35-1.42-1.26-1.8-1.68-1.22-1.33-3.26-2.99-5-3,.07-.84-.83-1.51-1-2-.71-2.1-1.01-2.96-2-5-.17-.34-.84-.64-1-1-1.37-3.03-1.81-5.38-3-8-.13-.28.13-.72,0-1-.16-.36-.84-.66-1-1-.6-1.29-1.67-2.29-2-3-.5-1.08-.63-2.19-1-3-.32-.71-.69-1.33-1-2-1.31-2.78-2.82-5.47-4-8-3.68-7.9-1.96-9.96,3.99-3v-6c-3.1,2.6-5.65-.62-7.49-.64-1.2-.01-3.1,1.8-5.96,1.71-2.19-.07-5.86-1.52-8.54-2.06-.61-.12-1.41.13-2,0-1.24-.28-2.82-1.79-4-2-1.47-.27-3.43.16-5.01,0-.05.66.05,1.34,0,2-.07.99.05,2.01,0,3-4.84,2.53-7.29-1.94-12-2-.78-1.34-2.53-3.63-4-4-1.24-.31-3.48.11-5,0-.33-.02-.67.01-1,0-.44-.01-1.64-1.14-2-1-4.29-3.98.64-6.53-1.01-11-1.47-.63-2.58.93-2.99,1-.89.16-2.03-.09-3,0-1.06-1.71-3.18-.9-5-1-.33-.02-.67,0-1,0-4.6.09-5.81.36-10,2-2.55,1-3.58,2.27-5,3-1,.52-1.81.99-3,1-.47,0-1.47-.98-2-1-.32-.01-.82-.08-1,0-4.75-.05-5.07-2.77-6,3-3.53,1.87-6.31,2.34-7.42,6.9-.88,3.61,1.53,2.84-2.58,7.1-1.25,1.29-2.71,2.74-4,4-4.89,4.79-7.31,6.62-10,14-.94,2.59.91,1.91,1,3,.21,2.62.22,6.38,0,9-.08.97.16,2.11,0,3-.11.6-3.95,4.42-.54,4.85,1.91.24,4.16-2.68,6.54,2.15-.3.41.29,2.41,0,3-.37.74-2.55.83-1.97,2.45.47,1.3,2.59,1.97,2.97,2.55.57.88.79,1.68,1,2,.28.43-.14,1.72.96,2.52.71.52,1.91-.56,2.04-.52.07.02.9.87,1,1,1.11,1.42.26,3.04,4,4,.26.07.67,0,1,0-.73,2.59,1.62,2.32,4,2,.48-.06,1,.08,1.48,0,.48-.18.58-.52.52-.99l-.18-.16c.1.07.17.12.18.16,1.38,0,2.76,0,4.13,0,.19.11.48.12.87,0,.98,0,2.03.09,3,0,4.47-.42,4.41-1.25,7-2,.65-.19,1.34-.85,2-1,.22.2.36.14.45-.07,1.02-.1,2.51.08,3.55.07.33,0,.67-.02,1,0-.24-.71-.26-1.37-.17-1.99.02.66.07,1.32.17,1.99,1.81.1,4.96-.15,6.42.9,2.35,1.7-2.91,4.11,6.58,2.1.6-.13,1.39.12,2,0,1.42,3.69,2.81,3.63,3,8,.04.83-1.44,3.19,0,5-1.75,1.98-1.91,6.98.5,6.98.41,0,1.83-2.45,3.26-2.87,1.29-.37,2.12.9,2.24.89-.13.57.2,1.73,0,2-.29.41-1.74.79-1.99,1.5v1.49c.99.03,1.99,0,2.99,0-.1.97.14,2.06,0,3-.05.32.03.67,0,1-.02.18-1.13.96-1,2,.38,3.02,3.34,8.15,3.09,12.52-.24,4.25-4.78,4.61-4.09,12.48" className="cls-1" />

            {/* Africa */}
            <path d="M386,180c-3.09-2.53-2.78-2.08-5.25-4.69-.59-.63-2.28-.42-3.23-1.3-2.43-2.26-.74-5.36-.52-8.01,1.48-.44,3.4,1.11,4.4,1.04.54-.04,3.91-7.45,8.59-2.03-.03-1.87-1.22-2.99-.62-5.02.93-3.11,4.22-2.05,3.63-4.98-3.93.79-4.49-2.16-1-3.5-1.36-.65-4.6-2.85-2.99-4.5,1.19-1.22,8.01,3.42,6.99-1h-4.99c1.9-4.47-2.43-6.06-2.08-10.43,1.4-.32,4.36-2.44,5.08-2.57,2.58-.47,6.75,2.49,6.01-.49-.17-.7-3.53-.81-5.27-1.25-.21-.09-.44-.18-.73-.26-.2-.79-2.5-4.42-3-5,.76-2.56.59-5.55,1-8,.19-1.1,1.89-3,1-5v-1c3.86-1.65,5.06-6.28,4.05-10.43-.29-1.18-1.58-.33-2.05-.57-2.64-1.32.47-6.97,2-9,7.7,2.8,10.94-1.57,1.28-1.67-.78,0-1.18.67-1.28.67-1.33.05-2.67-.02-4,0,.13-.82-.23-2.69,0-2.97.45-.55,11.07-1.48,12.99-2.53,1.85-1.02,1.61-3.24,2.62-4.38,1.83-2.08,4.72-4.02,6.34-6.61-1.95-2.7-7.18-6.76-10.03-3.08-1.15,1.49-.62,4.94-2.53,5.52-1.59.48-4.39-1.96-6.43-.51-.62.44-6.96,12.21-6.96,13.57-5.47,1.56-2.26-6.55-1-10,1.19-3.26,4.98-7.24,3.99-12l-9.99,10c-8.16.63-2.22-4.68,0-7,.48-.5,2.34-2.66,2-2.99-2.79-.71-4.26-.64-7-.01-5.07,1.16-7.58,5.09-11.99,5-1.71-6.62,3.89-5.38,5.99-9,.29-.5.09-1.6,1-2,.66.05,1.34-.03,2,0,2.63.12,7.61,1.13,9.51,1.02,2.13-.12,2.34-1.97,2.49-2.02,1.13-.41,2.79,0,4.28-.94,2.78-1.77,2.97-5.07-.31-6.54-3-1.35-6.96,1.39-5.98-3.52.38-.97,1.43-1.52.99-3-3.37,2.48-7.52,0-10.42.26-1.47.13-2.09,1.56-2.57,1.74-1.88.68-12.18,4.88-12.99,4-.28-.31-.07-3.23.46-4.03.94-1.43,4.61-1.18,5.36-2.22.5-.69-.03-2.5.2-2.71.62-.58,3.45.66,4.51.51,1.63-.23,2.83-1.6,4.93-1.59,3.09,0,2.59,1.4,6.54.05.58-.2,2.03.5,2-.5-1.66-1.99-3.6-2.56-6-1.5-1.11-.75-1.58-1.7-2-2-2.67-1.9-3.91-2.5-7.55-2.06-2.64.32-1.02,1.3-1.45,2.06-6.28-2.55-6.63,1.05-12,2.01,1.11,2.43,3.64.92,2.83,4.24-.19.77-2.82,6.33-3.11,6.54-.82.59-2.56.05-2.72.21-.75.74.69,6.46,1,7.99-3.41.15-4.52.75-4-3-6.39,3.27-2.66-1.96-4.92-3.57-1.05-.75-2.81-.36-4.07-.42.73,4.87,6.35,6.02,6,9.99h-5.01c-.6,2.24,2.64,6.71-.63,7.89-3.05,1.1-3.52-1.04-5.37-2.88.12-2.52-2-2.91-3-4-.31-.34-.67-.66-1-1-1.5-1.55-2.71-4.13-4.99-5l2.99,8c-.76,3" className="cls-1" />
            
            {/* Europe */}
            <path d="M897,206.01c-2.73-.57-3.53-.4-4.11-1.14-.15-.49-.43-1.1-.88-1.88-1.7-4.4-1.94-3.92-5-1-.89.85-1.4,2.39-2.69,2.82-1.29.42-4.11-.02-4.32.18-.92.88,2.67,5.51-5,4.01,1.87-6.13-3.54-11.11-9.04-12.61-2.37-.65-2.45.53-2.95.61-.64.1-1.41-.12-2,0-.75.16-1.87,1.4-2.89.93-.2-.09-.52-1.47-1.11-1.93-4.7-3.69-2.16-.17-2,1,.6,4.37-.85,7.74-5.75,5.71-.49-.2-3.15-4.41-3.4-4.53-.47-.22-2.09.96-3.23.74-2.89-.55-6.87-2.81-9.21-1.01-1.37,1.05-.85,4.26-3.81,3-4.35-1.86-2.86-8.98-4.07-10.97-.58-.96-2.13-.66-2.53-.93-.53-.36-.48-1.73-2.51-2.02-2.46-.35-4.31.67-6.49,1.02-.93.15-1.95,1.13-3.51,1.08-3.67-.12-2.85-2.18-4.59-3.48-1.32-.99-4.96-2.78-5.47-3.61-1.06-1.72.7-3.84.57-3.98-.09-.1-2.39-.91-2.52-.91-.99,0-3.29,6.85-6.97,5.91-2.5-.64-.37-6.85-1.63-7.89-.25-.2-5.86-1.1-6.28-1.08-2.92.09-3.41,9.45-3.78,9.85-.31.34-1.87-.08-2.48.4-1.04.81-.75,2.77-2.82,2.79-2.15.03-4.16-3.96-5.03-4.01-.6-.04-1.2,1.81-1.48,1.94-.56.26-2.16-.18-3,0-.25-.26-2.59-.25-2.49-2-1.81.69-1.56,5.1-2.44,6.41-2.36,3.54-10.17-7.38-9.08-10.41-1.68-.44-.66,1.5-1.1,1.82-1.88,1.34-3.87-1.39-4.88-2.82-.03-.72-.08-1.32-.16-1.85.56-3.09,5.16-5.23,5.16-5.63-.01-3.58-6.37-2.3-9-3.51l4,2c1.5,2.77-2,4.7-2.01,5-6.58-.13-10.2.06-16-3,1.77-4.32-6.42-5.08-7.94-3.9-.38.3.12,1.64-.06,1.9-4.61,6.83-4.68-2.68-5.01-3.01-1.31-1.31-8.35,3.39-10.15,4.4-1.05.59-2.74.33-3.57.88-.53.36-3.58,6.25-6.27,3.72-3-2.82,17.76-21.5,19.55-23.03,1.26-2.57-4.48-6.23.45-7.46-5.05-10.13-7.16-5.15-15.99-4.51-.73-1.39,1.2-4.83.99-4.99-1.04-.8-7.43,1.01-6-4.01.26-.13.72.13,1,0,3.22-1.45,1.68-3.98-2-2-3.51,1.88-6.05,6.62-3,10l-1,2c-1.26-.33-2.65-.06-3.99-1-.41.51,1.95,6.05.99,7-1,.98-5.54.14-4-3.01-2.75,1.6-8.16,4.16-9.49.01-2.68,4.08-7.46,4.02-11.05,5.95-7.13,3.84-7.55,7.83-5.45,15.04l-9.01,2c-1.9-.07-6.49-.53-7.58.97-.35.49.54,6.57.77,6.87.38.5,1.69.03,1.8.16,1.62,1.79,2.75,4.49,5,6-1.09,4.96.28,8.34-1,12.99.89.18,2.36-.25,3,0,.7.28,1.38,2.5,1,2.99-1.88,2.23-8.11-2.04-8.79-3.2-1.86-3.18,1.29-11.46.78-11.78-.47-.3-2.61.46-3.8-.21-.62-.35" className="cls-1" />
            
            {/* South America */}
            <path d="M352.96,443.1c-.3-.05-.62-.08-.96-.1-1.09-.42-2.83-3.7-5.78-4.71-3.66-1.26-11.69.04-12.14-.37-.46-.43.26-4.22-3.82-4.78-3.23-.44-3.55,2.79-5.99,3.64-3.67,1.27-7.43-2.05-11.27-.77l-1-2c1.36-.78,7.63-5.63,7.92-6.62.67-2.32-1.18-3.12-1.92-4.38-.7-1.19-1.38-1.33-2-2-.96-1.03.87-1.42-3-3-3.36-1.37-4.2-.54-7-1-.63-.1-1.36.09-2,0-6.34-.92-2.64-4.56-6-6-.5-.21-1.69.22-2,0-1.3-.95-1.52-3.05-4.48-4.11-2.47-.88-8.93-.08-12.78-1.12-1.31-.35-3.78-4.16-5.64-1.67-1.34,1.8,2.63,6.19-4.09,4.89-.28-1.47.21-3.43,0-5.01-.66.02-1.33-.03-1.99,0-1.06.06-2.09.49-3,1-.36.2-.67.69-1,1-1.6,1.53-2.31,3.28-3.61,4.95-.57.72-2.12.99-2.22,1.19-.21.41,1.96,7.05,1.62,9.13-.35,2.15-3.46,3.89-3.78,4.73-.18.46.12,1.39,0,2-2.01-.73-5.13,3.74-5.05,5.48.04.82,3.32-1.02,1.05,5.52-.19.55.21,1.79,0,2-.19.18-1.65-.26-1.9.07-.65.84,4.74,7.17,5.9,8.93l3,6c1.31,2.87,2.4,6.83,4.49,9.01,2.04,2.12,9.92,6.51,12.51,5.99.6-.12,1.36.08,2,0,.01,1,0,2,0,3-.33.08-.79-.12-1,0-1.93,1.12-.81,6.72-1,9-.53,6.33-1.21,12.7-2,19-.39,3.11-.52,6.84-1,10-1.26,8.23-3.94,11.86-4.1,20.39-.05,2.51-2.86,6.27,1.1,5.61h1c0,.21.79,2.46,1,4,.23,1.74.62,3.27,1,5v5c-.65-.06-1.41.12-2,0,.26-2.22-.17-4.73-.01-7h-.98c-.28,1.85.35,4.29-.01,6-.25,1.17-1.64,3.36-2,5-.59,2.66.02,2.27,0,3-.05,2.08-.3,5.1,0,7,.49,3.13,2.23,2.23,2.97,3.79.58,1.22.67,7.24,1.54,7.22,4.25-.66,2.99-4.54,3.74-7.28.4-1.46,1.36-2.52,1.75-3.73.65-1.41,6-6.06,5.98-6.5-.01-.49-6.51-3.64-3.35-8.03.12-.15.24-.3.37-.46,8.27-2.13,4.38-14.36,5-15,11.2,1.12,5.28-1.69,7-7,3.79,1.08,5.6.65,9-1-.15.03-.32.04-.5.06l.5-.06c1.43-.69,3.14-.64,3.5-3.49.6-4.85-3.48-3.94-3.5-9.51.62.11,1.5-.16,2,0,4.13,1.34,7.96,5.01,10-1,.25-.75-.26-2.26,0-3,.19-.55.76-.71,1-1,2.37-2.87,9.27-8.57,10.22-11.28,1.19-3.38.02-6.03,2.2-8.8,3.44-4.39,12.28-3.98,15.14-6.86,3.46-3.48,3.69-6.86,4.44-11.06.63-3.56,1.62-7.06,1-11,1.31-2.39,5.85-6.47,8-10,1.12-1.83,3.26-2.27,1.87-6.36-.68-2-2.55-1.58-3.91-1.53" className="cls-1" />
            
            {/* Australia */}
            <path d="M670.81,104.61c1.37,5.38,2.91,3.05,5.5,4.62,1.77,1.07,1.38,3.69,4.18,2.76,2.96-.98-.54-3.04-.51-4.48.04-1.6,3.55-5.3,2.01-8.5-2.07,1.09-3.12,3.69-5.99,2.99l.99-4.99c-4.81-.23-7.37,2.97-6.19,7.6Z" className="cls-1" />
            
            <style jsx>{`
              .cls-1 {
                fill: #e5e7eb;
                stroke: #f9fafb;
                stroke-width: 0.5px;
                stroke-miterlimit: 10;
                opacity: 0.8;
              }
              .dark .cls-1 {
                fill: #374151;
                stroke: #4b5563;
              }
            `}</style>
          </g>
          
          {/* Visitor location dots */}
          {locations.map((location, index) => {
            const { x, y } = latLngToSVG(location.latitude, location.longitude);
            const dotSize = getDotSize(location.count, maxCount);
            const dotColor = getDotColor(location.count, maxCount);
            
            return (
              <g key={index}>
                {/* Outer glow for larger dots */}
                {location.count > maxCount * 0.5 && (
                  <circle
                    cx={x}
                    cy={y}
                    r={dotSize + 2}
                    fill={dotColor}
                    fillOpacity="0.3"
                    className="animate-pulse"
                  />
                )}
                
                {/* Main dot */}
                <circle
                  cx={x}
                  cy={y}
                  r={dotSize}
                  fill={dotColor}
                  stroke="white"
                  strokeWidth="1"
                  className="cursor-pointer hover:stroke-2 transition-all duration-200"
                  onMouseEnter={() => setHoveredLocation(location)}
                  onMouseLeave={() => setHoveredLocation(null)}
                  onClick={() => setSelectedLocation(
                    selectedLocation?.latitude === location.latitude && 
                    selectedLocation?.longitude === location.longitude 
                      ? null 
                      : location
                  )}
                  data-testid={`location-dot-${location.countryCode || index}`}
                />
              </g>
            );
          })}
          
          {/* Grid lines for reference */}
          <defs>
            <pattern id="grid" width="100" height="50" patternUnits="userSpaceOnUse">
              <path d="M 100 0 L 0 0 0 50" fill="none" stroke="#e5e7eb" strokeWidth="0.5" opacity="0.3" />
            </pattern>
          </defs>
          <rect width="1000" height="500" fill="url(#grid)" />
        </svg>

        {/* Tooltip */}
        {(hoveredLocation || selectedLocation) && (
          <div 
            className="absolute z-10 bg-white dark:bg-gray-800 border border-gray-200 dark:border-gray-600 rounded-lg shadow-lg p-3 pointer-events-none"
            style={{
              left: '50%',
              top: '10px',
              transform: 'translateX(-50%)'
            }}
            data-testid="location-tooltip"
          >
            <div className="text-sm font-medium text-gray-900 dark:text-gray-100">
              {(hoveredLocation || selectedLocation)?.city && (
                <span>{(hoveredLocation || selectedLocation)?.city}, </span>
              )}
              {(hoveredLocation || selectedLocation)?.country}
            </div>
            <div className="text-xs text-gray-600 dark:text-gray-400 mt-1">
              <div>{(hoveredLocation || selectedLocation)?.sessions} sessions</div>
              <div>{(hoveredLocation || selectedLocation)?.count} page views</div>
            </div>
          </div>
        )}
      </div>

      {/* Legend */}
      <div className="mt-4 flex items-center justify-between text-xs text-gray-600 dark:text-gray-400">
        <div className="flex items-center space-x-4">
          <div className="flex items-center space-x-1">
            <div className="w-2 h-2 rounded-full bg-green-500"></div>
            <span>Low activity</span>
          </div>
          <div className="flex items-center space-x-1">
            <div className="w-2 h-2 rounded-full bg-yellow-500"></div>
            <span>Medium</span>
          </div>
          <div className="flex items-center space-x-1">
            <div className="w-2 h-2 rounded-full bg-red-500"></div>
            <span>High activity</span>
          </div>
        </div>
        
        {locationsData?.summary && (
          <div className="text-right">
            <div>{locationsData.summary.totalLocations} locations</div>
            <div>{locationsData.summary.totalCountries} countries</div>
          </div>
        )}
      </div>
    </div>
  );
}