import { useQuery } from "@tanstack/react-query";
import { LineChart, Line, XAxis, YAxis, CartesianGrid, Tooltip, ResponsiveContainer, BarChart, Bar } from 'recharts';
import { Calendar, TrendingUp, Users, Eye } from "lucide-react";
import { useState } from "react";

interface DailyStat {
  date: string;
  uniqueVisitors: number;
  totalPageViews: number;
  countries: Record<string, number>;
  topPages: Record<string, number>;
}

interface VisitorStatsResponse {
  dailyStats: DailyStat[];
  totalToday: number;
  totalThisWeek: number;
  dateRange: {
    start: string;
    end: string;
  };
}

interface VisitorChartProps {
  className?: string;
}

// Format date for display
function formatDate(dateStr: string): string {
  const date = new Date(dateStr);
  return date.toLocaleDateString('en-US', { 
    month: 'short', 
    day: 'numeric' 
  });
}

// Custom tooltip for the chart
function CustomTooltip({ active, payload, label }: any) {
  if (active && payload && payload.length) {
    const data = payload[0].payload;
    return (
      <div className="bg-white dark:bg-gray-800 border border-gray-200 dark:border-gray-600 rounded-lg shadow-lg p-3">
        <p className="text-sm font-medium text-gray-900 dark:text-gray-100">
          {formatDate(label)}
        </p>
        <div className="mt-1 space-y-1">
          <div className="flex items-center text-xs">
            <Users className="h-3 w-3 text-blue-500 mr-1" />
            <span className="text-gray-600 dark:text-gray-400">Visitors: </span>
            <span className="font-medium text-blue-600 dark:text-blue-400 ml-1">
              {data.uniqueVisitors}
            </span>
          </div>
          <div className="flex items-center text-xs">
            <Eye className="h-3 w-3 text-green-500 mr-1" />
            <span className="text-gray-600 dark:text-gray-400">Page views: </span>
            <span className="font-medium text-green-600 dark:text-green-400 ml-1">
              {data.totalPageViews}
            </span>
          </div>
        </div>
      </div>
    );
  }
  return null;
}

export function VisitorChart({ className = "" }: VisitorChartProps) {
  const [chartType, setChartType] = useState<'line' | 'bar'>('line');
  const [dataType, setDataType] = useState<'visitors' | 'pageviews'>('visitors');

  const { data: statsData, isLoading, error } = useQuery<VisitorStatsResponse>({
    queryKey: ['/api/admin/visitor-stats'],
    refetchInterval: 5 * 60 * 1000, // Refresh every 5 minutes
  });

  if (isLoading) {
    return (
      <div className={`flex items-center justify-center h-64 bg-gray-50 dark:bg-gray-800 rounded-lg ${className}`}>
        <div className="text-center">
          <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600 mx-auto mb-2" />
          <p className="text-sm text-gray-600 dark:text-gray-400">Loading visitor data...</p>
        </div>
      </div>
    );
  }

  if (error) {
    return (
      <div className={`flex items-center justify-center h-64 bg-gray-50 dark:bg-gray-800 rounded-lg ${className}`}>
        <div className="text-center">
          <p className="text-sm text-red-600 dark:text-red-400">Failed to load visitor statistics</p>
          <p className="text-xs text-gray-500 dark:text-gray-500 mt-1">Chart unavailable</p>
        </div>
      </div>
    );
  }

  const chartData = statsData?.dailyStats.map(stat => ({
    date: stat.date,
    uniqueVisitors: stat.uniqueVisitors,
    totalPageViews: stat.totalPageViews,
    displayDate: formatDate(stat.date)
  })) || [];

  const totalVisitors = chartData.reduce((sum, day) => sum + day.uniqueVisitors, 0);
  const totalPageViews = chartData.reduce((sum, day) => sum + day.totalPageViews, 0);
  const avgDailyVisitors = Math.round(totalVisitors / Math.max(chartData.length, 1));

  return (
    <div className={`${className}`} data-testid="visitor-chart">
      <div className="bg-white dark:bg-gray-900 rounded-lg border border-gray-200 dark:border-gray-700 p-4">
        {/* Header with controls */}
        <div className="flex items-center justify-between mb-4">
          <div className="flex items-center space-x-2">
            <TrendingUp className="h-5 w-5 text-blue-600" />
            <h3 className="text-lg font-semibold text-gray-900 dark:text-gray-100">
              Visitor Trends
            </h3>
          </div>
          
          <div className="flex items-center space-x-2">
            {/* Data type toggle */}
            <div className="flex bg-gray-100 dark:bg-gray-800 rounded-lg p-1">
              <button
                onClick={() => setDataType('visitors')}
                className={`px-3 py-1 text-xs font-medium rounded-md transition-colors ${
                  dataType === 'visitors' 
                    ? 'bg-white dark:bg-gray-700 text-blue-600 dark:text-blue-400 shadow-sm' 
                    : 'text-gray-600 dark:text-gray-400 hover:text-gray-900 dark:hover:text-gray-200'
                }`}
                data-testid="toggle-visitors"
              >
                Visitors
              </button>
              <button
                onClick={() => setDataType('pageviews')}
                className={`px-3 py-1 text-xs font-medium rounded-md transition-colors ${
                  dataType === 'pageviews' 
                    ? 'bg-white dark:bg-gray-700 text-green-600 dark:text-green-400 shadow-sm' 
                    : 'text-gray-600 dark:text-gray-400 hover:text-gray-900 dark:hover:text-gray-200'
                }`}
                data-testid="toggle-pageviews"
              >
                Page Views
              </button>
            </div>

            {/* Chart type toggle */}
            <div className="flex bg-gray-100 dark:bg-gray-800 rounded-lg p-1">
              <button
                onClick={() => setChartType('line')}
                className={`px-2 py-1 text-xs font-medium rounded-md transition-colors ${
                  chartType === 'line' 
                    ? 'bg-white dark:bg-gray-700 text-gray-900 dark:text-gray-100 shadow-sm' 
                    : 'text-gray-600 dark:text-gray-400 hover:text-gray-900 dark:hover:text-gray-200'
                }`}
                data-testid="chart-type-line"
              >
                Line
              </button>
              <button
                onClick={() => setChartType('bar')}
                className={`px-2 py-1 text-xs font-medium rounded-md transition-colors ${
                  chartType === 'bar' 
                    ? 'bg-white dark:bg-gray-700 text-gray-900 dark:text-gray-100 shadow-sm' 
                    : 'text-gray-600 dark:text-gray-400 hover:text-gray-900 dark:hover:text-gray-200'
                }`}
                data-testid="chart-type-bar"
              >
                Bar
              </button>
            </div>
          </div>
        </div>

        {/* Summary stats */}
        <div className="grid grid-cols-2 md:grid-cols-4 gap-4 mb-6">
          <div className="text-center">
            <div className="text-2xl font-bold text-blue-600 dark:text-blue-400" data-testid="stat-today">
              {statsData?.totalToday || 0}
            </div>
            <div className="text-xs text-gray-600 dark:text-gray-400">Today</div>
          </div>
          <div className="text-center">
            <div className="text-2xl font-bold text-green-600 dark:text-green-400" data-testid="stat-week">
              {statsData?.totalThisWeek || 0}
            </div>
            <div className="text-xs text-gray-600 dark:text-gray-400">This Week</div>
          </div>
          <div className="text-center">
            <div className="text-2xl font-bold text-purple-600 dark:text-purple-400" data-testid="stat-total">
              {totalVisitors}
            </div>
            <div className="text-xs text-gray-600 dark:text-gray-400">30 Days</div>
          </div>
          <div className="text-center">
            <div className="text-2xl font-bold text-orange-600 dark:text-orange-400" data-testid="stat-average">
              {avgDailyVisitors}
            </div>
            <div className="text-xs text-gray-600 dark:text-gray-400">Daily Avg</div>
          </div>
        </div>

        {/* Chart */}
        <div className="h-64" data-testid="chart-container">
          <ResponsiveContainer width="100%" height="100%">
            {chartType === 'line' ? (
              <LineChart data={chartData} margin={{ top: 5, right: 30, left: 20, bottom: 5 }}>
                <CartesianGrid 
                  strokeDasharray="3 3" 
                  className="stroke-gray-200 dark:stroke-gray-700" 
                />
                <XAxis 
                  dataKey="displayDate"
                  className="text-gray-600 dark:text-gray-400"
                  fontSize={12}
                />
                <YAxis 
                  className="text-gray-600 dark:text-gray-400"
                  fontSize={12}
                />
                <Tooltip content={<CustomTooltip />} />
                <Line
                  type="monotone"
                  dataKey={dataType === 'visitors' ? 'uniqueVisitors' : 'totalPageViews'}
                  stroke={dataType === 'visitors' ? '#2563eb' : '#16a34a'}
                  strokeWidth={2}
                  dot={{ fill: dataType === 'visitors' ? '#2563eb' : '#16a34a', r: 3 }}
                  activeDot={{ r: 5 }}
                />
              </LineChart>
            ) : (
              <BarChart data={chartData} margin={{ top: 5, right: 30, left: 20, bottom: 5 }}>
                <CartesianGrid 
                  strokeDasharray="3 3" 
                  className="stroke-gray-200 dark:stroke-gray-700" 
                />
                <XAxis 
                  dataKey="displayDate"
                  className="text-gray-600 dark:text-gray-400"
                  fontSize={12}
                />
                <YAxis 
                  className="text-gray-600 dark:text-gray-400"
                  fontSize={12}
                />
                <Tooltip content={<CustomTooltip />} />
                <Bar
                  dataKey={dataType === 'visitors' ? 'uniqueVisitors' : 'totalPageViews'}
                  fill={dataType === 'visitors' ? '#2563eb' : '#16a34a'}
                  radius={[2, 2, 0, 0]}
                />
              </BarChart>
            )}
          </ResponsiveContainer>
        </div>

        {/* Footer */}
        <div className="flex items-center justify-between text-xs text-gray-500 dark:text-gray-400 mt-4">
          <div className="flex items-center space-x-1">
            <Calendar className="h-3 w-3" />
            <span>Last 30 days</span>
          </div>
          <div>
            Data updates every 5 minutes
          </div>
        </div>
      </div>
    </div>
  );
}