import { useState, useEffect, useRef } from 'react';
import { Brush, PenTool } from 'lucide-react';
import { Palette } from 'lucide-react';
import { logoComposerIcon } from '@/assets';

interface ToolbarState {
  x: number;
  y: number;
  dock: 'left' | 'right' | 'free';
  activeTab: 'logo' | 'text' | 'shapes' | 'color' | null;
  showPanel: boolean;
  collapsed: boolean;
  z: number;
  showLeftSnapHint: boolean;
  showRightSnapHint: boolean;
  tooltip: {
    show: boolean;
    text: string;
    buttonRef: HTMLButtonElement | null;
  };
}

interface ToolbarProps {
  logoEditorContent: React.ReactNode;
  textEditorContent: React.ReactNode;
  shapesEditorContent: React.ReactNode;
  colorEditorContent: React.ReactNode;
}

const STORAGE_KEY = 'ibrandbiz-toolbar-state';
const SNAP_THRESHOLD = 12;
const EDGE_PADDING = 8;
const LEFT_DOCK_PADDING = 20; // Dock positioning space of 20px

let zCounter = 30;

export function Toolbar({ logoEditorContent, textEditorContent, shapesEditorContent, colorEditorContent }: ToolbarProps) {
  const [state, setState] = useState<ToolbarState>(() => {
    // Load from localStorage or use defaults
    const stored = localStorage.getItem(STORAGE_KEY);
    if (stored) {
      try {
        const parsed = JSON.parse(stored);
        return {
          x: typeof parsed.x === 'number' ? Math.max(LEFT_DOCK_PADDING, parsed.x) : LEFT_DOCK_PADDING,
          y: parsed.y || 96,
          dock: parsed.dock || 'free',
          activeTab: parsed.activeTab || null,
          showPanel: false,
          collapsed: parsed.collapsed || false,
          z: ++zCounter,
          showLeftSnapHint: false,
          showRightSnapHint: false,
          tooltip: {
            show: false,
            text: '',
            buttonRef: null
          }
        };
      } catch (e) {
        console.warn('Failed to parse toolbar state from localStorage');
      }
    }
    
    return {
      x: LEFT_DOCK_PADDING,
      y: 96,
      dock: 'free',
      activeTab: null,
      showPanel: false,
      collapsed: false,
      z: ++zCounter,
      showLeftSnapHint: false,
      showRightSnapHint: false,
      tooltip: {
        show: false,
        text: '',
        buttonRef: null
      }
    };
  });

  const toolbarRef = useRef<HTMLDivElement>(null);
  const dragRef = useRef<{
    isDragging: boolean;
    startX: number;
    startY: number;
    originX: number;
    originY: number;
  }>({
    isDragging: false,
    startX: 0,
    startY: 0,
    originX: 0,
    originY: 0
  });
  const tooltipTimerRef = useRef<NodeJS.Timeout | null>(null);

  // Persist state to localStorage whenever it changes (excluding tooltip state)
  useEffect(() => {
    const { tooltip, ...persistableState } = state;
    localStorage.setItem(STORAGE_KEY, JSON.stringify(persistableState));
  }, [state.x, state.y, state.dock, state.activeTab, state.collapsed]);

  // Keyboard shortcuts and focus management
  useEffect(() => {
    const handleKeyDown = (e: KeyboardEvent) => {
      const isTypingInInput = e.target instanceof HTMLInputElement || e.target instanceof HTMLTextAreaElement;

      switch (e.key.toLowerCase()) {
        case 'escape':
          // ESC always works to close panels, even when typing in inputs
          // Keep activeTab so we remember which tab was last active
          if (state.showPanel) {
            setState(prev => ({ ...prev, showPanel: false }));
            e.preventDefault();
          }
          break;
        case 'l':
        case 't':
        case 's':
        case 'c':
          // Letter shortcuts only work when NOT typing in inputs
          if (!isTypingInInput) {
            const tabMap = { 'l': 'logo', 't': 'text', 's': 'shapes', 'c': 'color' } as const;
            togglePanel(tabMap[e.key.toLowerCase() as keyof typeof tabMap]);
            e.preventDefault();
          }
          break;
      }
    };

    // Add global keyboard listener
    document.addEventListener('keydown', handleKeyDown);
    return () => document.removeEventListener('keydown', handleKeyDown);
  }, [state.showPanel]);

  // Focus management - bring toolbar to front when clicked
  const handleToolbarClick = () => {
    setState(prev => ({ ...prev, z: ++zCounter }));
  };

  // Clamp position to viewport bounds
  const clampPosition = (x: number, y: number, width: number, height: number) => {
    const maxX = window.innerWidth - width - EDGE_PADDING;
    const maxY = window.innerHeight - height - EDGE_PADDING;
    
    return {
      x: Math.max(EDGE_PADDING, Math.min(x, maxX)),
      y: Math.max(EDGE_PADDING, Math.min(y, maxY))
    };
  };

  // Handle window resize - re-clamp position
  useEffect(() => {
    const handleResize = () => {
      if (!toolbarRef.current) return;
      
      const rect = toolbarRef.current.getBoundingClientRect();
      const { x: clampedX, y: clampedY } = clampPosition(state.x, state.y, rect.width, rect.height);
      
      if (clampedX !== state.x || clampedY !== state.y) {
        setState(prev => ({ ...prev, x: clampedX, y: clampedY }));
      }
    };

    window.addEventListener('resize', handleResize);
    return () => window.removeEventListener('resize', handleResize);
  }, [state.x, state.y]);

  // Drag handlers
  const handlePointerDown = (e: React.PointerEvent) => {
    e.preventDefault();
    const element = e.currentTarget as HTMLElement;
    element.setPointerCapture(e.pointerId);
    
    dragRef.current = {
      isDragging: true,
      startX: e.clientX,
      startY: e.clientY,
      originX: state.x,
      originY: state.y
    };

    // Bring to front
    setState(prev => ({ ...prev, z: ++zCounter }));
  };

  const handlePointerMove = (e: React.PointerEvent) => {
    if (!dragRef.current.isDragging || !toolbarRef.current) return;
    
    e.preventDefault();
    
    const dx = e.clientX - dragRef.current.startX;
    const dy = e.clientY - dragRef.current.startY;
    
    const newX = dragRef.current.originX + dx;
    const newY = dragRef.current.originY + dy;
    
    const rect = toolbarRef.current.getBoundingClientRect();
    const { x: clampedX, y: clampedY } = clampPosition(newX, newY, rect.width, rect.height);
    
    // Check snap proximity for visual hints
    const showLeftSnap = clampedX <= SNAP_THRESHOLD;
    const showRightSnap = clampedX >= window.innerWidth - rect.width - SNAP_THRESHOLD;
    
    setState(prev => ({ 
      ...prev, 
      x: clampedX, 
      y: clampedY,
      showLeftSnapHint: showLeftSnap,
      showRightSnapHint: showRightSnap
    }));
  };

  const handlePointerUp = (e: React.PointerEvent) => {
    if (!dragRef.current.isDragging || !toolbarRef.current) return;
    
    e.preventDefault();
    const element = e.currentTarget as HTMLElement;
    element.releasePointerCapture(e.pointerId);
    
    // Check for edge snapping
    const rect = toolbarRef.current.getBoundingClientRect();
    let newDock = state.dock;
    let snapX = state.x;
    
    if (state.x <= SNAP_THRESHOLD) {
      newDock = 'left';
      snapX = LEFT_DOCK_PADDING;
    } else if (state.x >= window.innerWidth - rect.width - SNAP_THRESHOLD) {
      newDock = 'right';
      snapX = window.innerWidth - rect.width - EDGE_PADDING;
    } else {
      newDock = 'free';
    }
    
    setState(prev => ({ 
      ...prev, 
      dock: newDock, 
      x: snapX,
      showLeftSnapHint: false,
      showRightSnapHint: false
    }));
    
    dragRef.current.isDragging = false;
  };

  // Helper to toggle panel for a specific tab
  const togglePanel = (tab: 'logo' | 'text' | 'shapes' | 'color') => {
    setState(prev => ({
      ...prev,
      activeTab: tab, // Always set activeTab to remember the last used tab
      showPanel: prev.activeTab === tab && prev.showPanel ? false : true
    }));
  };

  // Helper to toggle collapse
  const toggleCollapse = () => {
    setState(prev => ({ ...prev, collapsed: !prev.collapsed }));
  };

  // Tooltip helpers
  const showTooltip = (text: string, buttonElement: HTMLButtonElement) => {
    // Clear any existing timer
    if (tooltipTimerRef.current) {
      clearTimeout(tooltipTimerRef.current);
    }
    
    // Set timer to show tooltip after 300ms
    tooltipTimerRef.current = setTimeout(() => {
      setState(prev => ({
        ...prev,
        tooltip: {
          show: true,
          text,
          buttonRef: buttonElement
        }
      }));
    }, 300);
  };

  const hideTooltip = () => {
    // Clear timer if hovering stopped before tooltip showed
    if (tooltipTimerRef.current) {
      clearTimeout(tooltipTimerRef.current);
      tooltipTimerRef.current = null;
    }
    
    // Hide tooltip immediately
    setState(prev => ({
      ...prev,
      tooltip: {
        show: false,
        text: '',
        buttonRef: null
      }
    }));
  };

  // Cleanup tooltip timer on unmount
  useEffect(() => {
    return () => {
      if (tooltipTimerRef.current) {
        clearTimeout(tooltipTimerRef.current);
      }
    };
  }, []);

  // Get tab content
  const getActiveContent = () => {
    switch (state.activeTab) {
      case 'logo': return logoEditorContent;
      case 'text': return textEditorContent;
      case 'shapes': return shapesEditorContent;
      case 'color': return colorEditorContent;
      default: return null;
    }
  };

  const buttonBarWidth = state.collapsed ? 60 : 140; // Wider when expanded to show text
  const panelWidth = state.activeTab === 'text' ? 384 : 320; // Text editor needs more width

  return (
    <>
      {/* Vertical Button Bar */}
      <div
        ref={toolbarRef}
        className="fixed bg-white border border-[#30342d] shadow-lg select-none"
        style={{
          transform: `translate3d(${state.x}px, ${state.y}px, 0)`,
          zIndex: state.z,
          width: `${buttonBarWidth}px`,
          pointerEvents: 'auto',
          transition: dragRef.current.isDragging ? 'none' : 'width 200ms ease, height 200ms ease',
          borderWidth: '0.25pt'
        }}
        onClick={handleToolbarClick}
      >
        {/* Header */}
        <div className="flex items-center justify-between p-2 border-b border-[#30342d] bg-[#30342d]">
          <div className="flex items-center gap-2">
            {/* Drag Grip Affordance */}
            <div 
              className="cursor-move text-white/60 hover:text-white transition-colors"
              onPointerDown={handlePointerDown}
              onPointerMove={handlePointerMove}
              onPointerUp={handlePointerUp}
              title="Drag to move toolbar"
            >
              <svg className="w-4 h-4" fill="currentColor" viewBox="0 0 24 24">
                <circle cx="5" cy="6" r="1.5"/>
                <circle cx="5" cy="12" r="1.5"/>
                <circle cx="5" cy="18" r="1.5"/>
                <circle cx="12" cy="6" r="1.5"/>
                <circle cx="12" cy="12" r="1.5"/>
                <circle cx="12" cy="18" r="1.5"/>
              </svg>
            </div>
            
            {!state.collapsed && (
              <span className="text-xs font-medium text-white">Tools</span>
            )}
          </div>
          
          <button
            onClick={toggleCollapse}
            className="p-1 hover:bg-orange-600 rounded transition-colors"
            title={state.collapsed ? 'Expand' : 'Collapse'}
          >
            <svg className="w-4 h-4 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
              {state.collapsed ? (
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 5l7 7-7 7" />
              ) : (
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M15 19l-7-7 7-7" />
              )}
            </svg>
          </button>
        </div>

        {/* Vertical Buttons */}
        <div className="p-2 space-y-2">
          <button
            onClick={() => togglePanel('logo')}
            onMouseEnter={(e) => showTooltip('Adjust logo size, rotation, and base color.', e.currentTarget)}
            onMouseLeave={hideTooltip}
            className={`w-full p-3 rounded-lg transition-colors relative ${
              state.activeTab === 'logo' && state.showPanel 
                ? 'bg-green-500 text-white' 
                : 'bg-gray-100 text-gray-700 hover:bg-gray-200'
            } ${state.collapsed ? 'flex items-center justify-center' : 'flex items-center gap-2'}`}
            title="Logo Editor (L)"
            tabIndex={0}
          >
            <img 
              src={logoComposerIcon} 
              alt="Logo Creator" 
              className={`w-5 h-5 ${state.collapsed ? '' : 'flex-shrink-0'}`}
            />
            {!state.collapsed && <span className="text-sm font-medium">Logo</span>}
            {/* Keyboard shortcut indicator */}
            <span className="absolute top-1 right-1 text-xs opacity-60 bg-black bg-opacity-10 rounded px-1">L</span>
          </button>
          
          <button
            onClick={() => togglePanel('text')}
            onMouseEnter={(e) => showTooltip('Edit fonts, styles, and alignment for brand text.', e.currentTarget)}
            onMouseLeave={hideTooltip}
            className={`w-full p-3 rounded-lg transition-colors relative ${
              state.activeTab === 'text' && state.showPanel 
                ? 'bg-green-500 text-white' 
                : 'bg-gray-100 text-gray-700 hover:bg-gray-200'
            } ${state.collapsed ? 'flex items-center justify-center' : 'flex items-center gap-2'}`}
            title="Text Editor (T)"
            tabIndex={0}
          >
            <PenTool className={`w-5 h-5 ${state.collapsed ? '' : 'flex-shrink-0'}`} />
            {!state.collapsed && <span className="text-sm font-medium">Text</span>}
            {/* Keyboard shortcut indicator */}
            <span className="absolute top-1 right-1 text-xs opacity-60 bg-black bg-opacity-10 rounded px-1">T</span>
          </button>
          
          <button
            onClick={() => togglePanel('shapes')}
            onMouseEnter={(e) => showTooltip('Add, resize, and rotate basic shapes.', e.currentTarget)}
            onMouseLeave={hideTooltip}
            className={`w-full p-3 rounded-lg transition-colors relative ${
              state.activeTab === 'shapes' && state.showPanel 
                ? 'bg-green-500 text-white' 
                : 'bg-gray-100 text-gray-700 hover:bg-gray-200'
            } ${state.collapsed ? 'flex items-center justify-center' : 'flex items-center gap-2'}`}
            title="Shapes Editor (S)"
            tabIndex={0}
          >
            <svg className={`w-5 h-5 ${state.collapsed ? '' : 'flex-shrink-0'}`} fill="none" stroke="currentColor" viewBox="0 0 24 24">
              <circle cx="7" cy="7" r="3" strokeWidth="2"/>
              <rect x="14" y="4" width="6" height="6" strokeWidth="2"/>
              <polygon points="17,13 20,19 14,19" strokeWidth="2"/>
            </svg>
            {!state.collapsed && <span className="text-sm font-medium">Shapes</span>}
            {/* Keyboard shortcut indicator */}
            <span className="absolute top-1 right-1 text-xs opacity-60 bg-black bg-opacity-10 rounded px-1">S</span>
          </button>
          
          <button
            onClick={() => togglePanel('color')}
            onMouseEnter={(e) => showTooltip('Choose colors for logo, text, and shapes.', e.currentTarget)}
            onMouseLeave={hideTooltip}
            className={`w-full p-3 rounded-lg transition-colors relative ${
              state.activeTab === 'color' && state.showPanel 
                ? 'bg-green-500 text-white' 
                : 'bg-gray-100 text-gray-700 hover:bg-gray-200'
            } ${state.collapsed ? 'flex items-center justify-center' : 'flex items-center gap-2'}`}
            title="Color Editor (C)"
            tabIndex={0}
          >
            <Palette className={`w-5 h-5 ${state.collapsed ? '' : 'flex-shrink-0'}`} />
            {!state.collapsed && <span className="text-sm font-medium">Color</span>}
            {/* Keyboard shortcut indicator */}
            <span className="absolute top-1 right-1 text-xs opacity-60 bg-black bg-opacity-10 rounded px-1">C</span>
          </button>
        </div>
      </div>

      {/* Popout Panel */}
      {state.showPanel && state.activeTab && (
        <div
          className="fixed bg-white border border-[#30342d] shadow-lg select-none"
          style={{
            transform: `translate3d(${state.x + buttonBarWidth + 8}px, ${state.y}px, 0)`,
            zIndex: state.z + 1,
            width: `${panelWidth}px`,
            maxHeight: '70vh',
            pointerEvents: 'auto',
            borderWidth: '0.25pt'
          }}
        >
          {/* Panel Header */}
          <div className="flex items-center justify-between p-3 border-b border-gray-200">
            <span className="text-sm font-medium text-gray-700">
              {state.activeTab === 'logo' && 'Logo Editor'}
              {state.activeTab === 'text' && 'Text Editor'}
              {state.activeTab === 'shapes' && 'Shapes Editor'}
              {state.activeTab === 'color' && 'Color Editor'}
            </span>
            
            <button
              onClick={() => setState(prev => ({ ...prev, showPanel: false }))}
              className="p-1 hover:bg-gray-100 rounded transition-colors"
              title="Close panel"
            >
              <svg className="w-4 h-4 text-gray-500" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M6 18L18 6M6 6l12 12" />
              </svg>
            </button>
          </div>
          
          {/* Panel Content */}
          <div className="overflow-y-auto p-4" style={{ maxHeight: 'calc(70vh - 60px)' }}>
            {getActiveContent()}
          </div>
        </div>
      )}

      {/* Tooltip */}
      {state.tooltip.show && state.tooltip.buttonRef && (
        (() => {
          const buttonRect = state.tooltip.buttonRef.getBoundingClientRect();
          const tooltipLeft = buttonRect.right + 8; // 8px gap from button
          const tooltipTop = buttonRect.top + (buttonRect.height / 2); // Center vertically on button
          
          return (
            <div
              className="fixed pointer-events-none"
              style={{
                left: `${tooltipLeft}px`,
                top: `${tooltipTop}px`,
                transform: 'translateY(-50%)',
                zIndex: state.z + 10
              }}
            >
              <div className="bg-gray-900 text-white text-xs px-2 py-1 rounded shadow-lg whitespace-nowrap">
                {state.tooltip.text}
              </div>
            </div>
          );
        })()
      )}

      {/* Snap Hints */}
      {state.showLeftSnapHint && (
        <div
          className="fixed top-0 left-0 h-full"
          style={{
            width: '12px',
            background: 'linear-gradient(to right, rgba(59, 130, 246, 0.2), transparent)',
            pointerEvents: 'none',
            zIndex: state.z - 1
          }}
        />
      )}
      
      {state.showRightSnapHint && (
        <div
          className="fixed top-0 right-0 h-full"
          style={{
            width: '12px',
            background: 'linear-gradient(to left, rgba(59, 130, 246, 0.2), transparent)',
            pointerEvents: 'none',
            zIndex: state.z - 1
          }}
        />
      )}
    </>
  );
}