import * as React from "react";
import { createPortal } from "react-dom";
import { X } from "lucide-react";
import { Button } from "@/components/ui/button";
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogFooter,
  DialogHeader,
  DialogTitle,
  DialogClose,
} from "@/components/ui/dialog";
import { Label } from "@/components/ui/label";
import { Separator } from "@/components/ui/separator";
import { cn } from "@/lib/utils";

export type SurveyOption = {
  id: string;
  label: string;
};

export type SurveyPayload = {
  surveyId: string;
  selectedOptionId: string;
  selectedLabel: string;
  createdAt: string; // ISO
  userId?: string | null;
  planId?: string | null;
};

type SurveyModalProps = {
  open: boolean;
  onOpenChange: (v: boolean) => void;

  /** A unique ID for this survey (used for localStorage "already answered"). */
  surveyId: string;

  /** Main title & description */
  title: string;
  description?: string;

  /** Radio options */
  options: SurveyOption[];

  /** Called when user submits (send to Firestore/DB/analytics here). */
  onSubmit?: (payload: SurveyPayload) => Promise<void> | void;

  /** Optional: pass current user/plan for analytics */
  userId?: string | null;
  planId?: string | null;

  /** Optional CSS class for the wrapper */
  className?: string;
};

export function SurveyModal(props: SurveyModalProps) {
  const {
    open,
    onOpenChange,
    surveyId,
    title,
    description,
    options,
    onSubmit,
    userId = null,
    planId = null,
    className,
  } = props;

  const [selected, setSelected] = React.useState<string>("");
  const [submitting, setSubmitting] = React.useState(false);
  const [error, setError] = React.useState<string | null>(null);

  React.useEffect(() => {
    if (!open) {
      setSelected("");
      setError(null);
    }
  }, [open]);

  const handleSubmit = async () => {
    setError(null);
    if (!selected) {
      setError("Please select an option to continue.");
      return;
    }
    try {
      setSubmitting(true);

      const chosen = options.find((o) => o.id === selected)!;
      const payload: SurveyPayload = {
        surveyId,
        selectedOptionId: chosen.id,
        selectedLabel: chosen.label,
        createdAt: new Date().toISOString(),
        userId,
        planId,
      };

      // Persist: local "answered" flag
      try {
        const key = `survey:${surveyId}:answered`;
        localStorage.setItem(key, "true");
      } catch {}

      // Callback to client code (DB, analytics, etc.)
      if (onSubmit) {
        await onSubmit(payload);
      }

      onOpenChange(false);
    } catch (e: any) {
      setError(e?.message || "Something went wrong. Please try again.");
    } finally {
      setSubmitting(false);
    }
  };

  return createPortal(
    <Dialog open={open} onOpenChange={onOpenChange}>
      <DialogContent className={cn("max-w-lg", className)} aria-describedby={undefined}>
        <DialogHeader>
          <DialogTitle>{title}</DialogTitle>
          {description ? (
            <DialogDescription>{description}</DialogDescription>
          ) : null}
        </DialogHeader>

        <div className="space-y-4">
          <Separator />
          <fieldset className="space-y-3">
            {options.map((opt) => (
              <label
                key={opt.id}
                className={cn(
                  "flex items-center gap-3 rounded-md border p-3 cursor-pointer hover:bg-muted",
                  selected === opt.id ? "border-primary" : "border-border"
                )}
              >
                <input
                  type="radio"
                  name={`survey-${surveyId}`}
                  className="h-4 w-4"
                  value={opt.id}
                  checked={selected === opt.id}
                  onChange={() => setSelected(opt.id)}
                  aria-labelledby={`survey-label-${opt.id}`}
                />
                <Label id={`survey-label-${opt.id}`} className="font-normal cursor-pointer">
                  {opt.label}
                </Label>
              </label>
            ))}
          </fieldset>

          {error ? (
            <p className="text-sm text-destructive" role="alert">
              {error}
            </p>
          ) : null}
        </div>

        <DialogFooter className="mt-2">
          <DialogClose asChild>
            <Button variant="ghost">
              <X className="h-4 w-4 mr-2" />
              Close
            </Button>
          </DialogClose>
          <Button onClick={handleSubmit} disabled={submitting}>
            {submitting ? "Submitting..." : "Submit"}
          </Button>
        </DialogFooter>
      </DialogContent>
    </Dialog>,
    document.body
  );
}

/** Utility to check if a survey has already been answered on this device. */
export function hasAnsweredSurvey(surveyId: string): boolean {
  try {
    return localStorage.getItem(`survey:${surveyId}:answered`) === "true";
  } catch {
    return false;
  }
}