import React, { useState, useEffect } from 'react';
import { loadStripe, Stripe } from '@stripe/stripe-js';
import {
  Elements,
  PaymentElement,
  useStripe,
  useElements
} from '@stripe/react-stripe-js';
import { useAuth } from '@/contexts/AuthContext';
import { useLocation } from 'wouter';
import { useToast } from '@/hooks/use-toast';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Loader2 } from 'lucide-react';

// Initialize Stripe promise
const stripePromise = loadStripe(import.meta.env.VITE_STRIPE_PUBLIC_KEY || '');

interface SubscribeElementsProps {
  priceType: 'monthly' | 'yearly';
  onSuccess?: () => void;
  onCancel?: () => void;
}

interface CheckoutFormProps {
  priceType: 'monthly' | 'yearly';
  onSuccess?: () => void;
  onCancel?: () => void;
}

function CheckoutForm({ priceType, onSuccess, onCancel }: CheckoutFormProps) {
  const stripe = useStripe();
  const elements = useElements();
  const { currentUser } = useAuth();
  const [, setLocation] = useLocation();
  const { toast } = useToast();
  
  const [isProcessing, setIsProcessing] = useState(false);
  const [errorMessage, setErrorMessage] = useState<string | null>(null);

  const handleSubmit = async (event: React.FormEvent) => {
    event.preventDefault();

    if (!stripe || !elements || !currentUser) {
      toast({
        title: 'Error',
        description: 'Payment system not ready. Please try again.',
        variant: 'destructive'
      });
      return;
    }

    setIsProcessing(true);
    setErrorMessage(null);

    try {
      // Wait for elements to be ready before confirming payment
      const { error: submitError } = await elements.submit();
      if (submitError) {
        throw new Error(submitError.message);
      }

      // Confirm the payment
      const { error, paymentIntent } = await stripe.confirmPayment({
        elements,
        confirmParams: {
          return_url: `${window.location.origin}/dashboard?welcome=pro`,
        },
        redirect: 'if_required',
      });

      if (error) {
        console.error('Payment confirmation error:', error);
        setErrorMessage(error.message || 'Payment failed. Please try again.');
        toast({
          title: 'Payment Failed',
          description: error.message || 'Payment failed. Please try again.',
          variant: 'destructive'
        });
      } else if (paymentIntent && paymentIntent.status === 'succeeded') {
        console.log('Payment succeeded:', paymentIntent.id);
        toast({
          title: 'Payment Successful!',
          description: 'Welcome to Pro! Your subscription is now active.',
        });
        
        // Call success callback or redirect to dashboard
        if (onSuccess) {
          onSuccess();
        } else {
          setLocation('/dashboard?welcome=pro');
        }
      }
    } catch (err: any) {
      console.error('Payment processing error:', err);
      setErrorMessage('Payment processing failed. Please try again.');
      toast({
        title: 'Payment Error',
        description: 'Payment processing failed. Please try again.',
        variant: 'destructive'
      });
    } finally {
      setIsProcessing(false);
    }
  };

  return (
    <Card className="w-full max-w-md mx-auto">
      <CardHeader>
        <CardTitle className="text-center">
          Complete Your Pro Subscription
        </CardTitle>
        <CardDescription className="text-center">
          {priceType === 'monthly' ? '$19/month' : '$190/year'}
        </CardDescription>
      </CardHeader>
      <CardContent>
        <form onSubmit={handleSubmit} className="space-y-6">
          <div className="min-h-[200px]">
            <PaymentElement 
              options={{
                layout: 'tabs',
                business: {
                  name: 'IBrandBiz'
                }
              }}
            />
          </div>
          
          {errorMessage && (
            <div className="p-3 rounded-md bg-red-50 border border-red-200">
              <p className="text-sm text-red-600">{errorMessage}</p>
            </div>
          )}
          
          <div className="flex gap-3">
            {onCancel && (
              <Button
                type="button"
                variant="outline"
                onClick={onCancel}
                disabled={isProcessing}
                className="flex-1"
                data-testid="button-cancel-subscription"
              >
                Cancel
              </Button>
            )}
            <Button
              type="submit"
              disabled={!stripe || isProcessing}
              className="flex-1 bg-green-500 hover:bg-green-600 text-white"
              data-testid="button-confirm-subscription"
            >
              {isProcessing ? (
                <>
                  <Loader2 className="w-4 h-4 mr-2 animate-spin" />
                  Processing...
                </>
              ) : (
                `Subscribe ${priceType === 'monthly' ? 'Monthly' : 'Yearly'}`
              )}
            </Button>
          </div>
          
          <div className="text-xs text-gray-500 text-center">
            You'll be charged {priceType === 'monthly' ? '$19' : '$190'} immediately.
            Cancel anytime from your account settings.
          </div>
        </form>
      </CardContent>
    </Card>
  );
}

export default function SubscribeElements({ priceType, onSuccess, onCancel }: SubscribeElementsProps) {
  const { currentUser } = useAuth();
  const { toast } = useToast();
  const [, setLocation] = useLocation();
  
  const [clientSecret, setClientSecret] = useState<string | null>(null);
  const [isCreating, setIsCreating] = useState(false);
  const [error, setError] = useState<string | null>(null);

  // Create subscription when component mounts
  useEffect(() => {
    const createSubscription = async () => {
      if (!currentUser) {
        setError('Please sign in to continue');
        return;
      }

      setIsCreating(true);
      setError(null);

      try {
        const token = await currentUser.getIdToken();
        const response = await fetch('/api/stripe/elements/create-subscription', {
          method: 'POST',
          headers: {
            'Authorization': `Bearer ${token}`,
            'Content-Type': 'application/json'
          },
          body: JSON.stringify({ priceType })
        });

        if (!response.ok) {
          const text = await response.text();
          const isJson = response.headers.get('content-type')?.includes('application/json');
          const errorData = isJson ? JSON.parse(text) : { error: `HTTP ${response.status}` };
          throw new Error(errorData.error || `Server error: ${response.status}`);
        }

        const data = await response.json();
        setClientSecret(data.clientSecret);
        
        console.log(`✅ Subscription created with client secret for ${priceType} plan`);
      } catch (err: any) {
        console.error('Subscription creation error:', err);
        const errorMessage = err.message || 'Failed to initialize payment. Please try again.';
        setError(errorMessage);
        
        // Handle specific error cases
        if (err.message?.includes('already has an active subscription')) {
          toast({
            title: 'Already Subscribed',
            description: 'You already have an active Pro subscription!',
          });
          setLocation('/dashboard');
          return;
        }
        
        toast({
          title: 'Setup Failed',
          description: errorMessage,
          variant: 'destructive'
        });
      } finally {
        setIsCreating(false);
      }
    };

    createSubscription();
  }, [currentUser, priceType, toast, setLocation]);

  // Show loading state while creating subscription
  if (isCreating) {
    return (
      <Card className="w-full max-w-md mx-auto">
        <CardContent className="p-8">
          <div className="flex flex-col items-center justify-center space-y-4">
            <Loader2 className="w-8 h-8 animate-spin text-green-500" />
            <p className="text-center text-gray-600">
              Setting up your subscription...
            </p>
          </div>
        </CardContent>
      </Card>
    );
  }

  // Show error state
  if (error) {
    return (
      <Card className="w-full max-w-md mx-auto">
        <CardContent className="p-8">
          <div className="text-center space-y-4">
            <div className="text-red-500 text-sm">
              {error}
            </div>
            <Button
              onClick={() => window.location.reload()}
              variant="outline"
              data-testid="button-retry-subscription"
            >
              Try Again
            </Button>
          </div>
        </CardContent>
      </Card>
    );
  }

  // Show payment form when client secret is ready
  if (clientSecret) {
    const options = {
      clientSecret,
      appearance: {
        theme: 'stripe' as const,
        variables: {
          colorPrimary: '#22c55e', // Green color matching the app
          colorBackground: '#ffffff',
          colorText: '#1f2937',
          colorDanger: '#ef4444',
          fontFamily: 'Inter, sans-serif',
          spacingUnit: '4px',
          borderRadius: '8px'
        }
      }
    };

    return (
      <Elements stripe={stripePromise} options={options}>
        <CheckoutForm
          priceType={priceType}
          onSuccess={onSuccess}
          onCancel={onCancel}
        />
      </Elements>
    );
  }

  // Fallback state
  return (
    <Card className="w-full max-w-md mx-auto">
      <CardContent className="p-8">
        <div className="text-center text-gray-600">
          Initializing payment system...
        </div>
      </CardContent>
    </Card>
  );
}