import React, { useEffect, useState } from "react";
import { Card, CardContent } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Button } from "@/components/ui/button";
import { Download, ZoomIn, ShoppingCart } from "lucide-react";
import { useCart } from "@/contexts/CartContext";
import { useEntitlement } from "@/hooks/useEntitlement";
import { fileTypeFrom, toTenDigitCode } from "@/utils/assetDisplay";

type Asset = { 
  id: string; 
  name: string; 
  previewUrl: string; 
  mimeType?: string; 
  code?: string;
  tags?: string[];
  category?: string;
};

interface StockCardProps {
  asset: Asset;
  onImageClick: (asset: Asset) => void;
}

export default function StockCard({ asset, onImageClick }: StockCardProps) {
  const { addItem, open } = useCart();
  const ent = useEntitlement(asset.id);

  const [code, setCode] = useState(asset.code || "");
  useEffect(() => {
    let ok = true;
    (async () => {
      if (asset.code) return setCode(asset.code);
      const c = await toTenDigitCode(asset.id);
      if (ok) setCode(c);
    })();
    return () => { ok = false; };
  }, [asset.id, asset.code]);

  const type = fileTypeFrom(asset);
  const displayName = `IBrandBiz | #${code}`;

  const handlePurchase = () => {
    addItem({
      kind: "stock",
      assetId: asset.id,
      name: displayName,
      priceCents: 699,
      qty: 1,
      previewUrl: asset.previewUrl,
    });
    open();
  };

  const handleDownload = () => {
    window.location.href = `/api/stock/${asset.id}/download`;
  };

  const button = (() => {
    if (ent.loading) return <Button size="sm" disabled className="text-xs px-2 py-1 h-7">Checking…</Button>;
    if (ent.licensed) return (
      <Button onClick={handleDownload} size="sm" className="text-xs px-2 py-1 h-7" data-testid={`download-${asset.id}`}>
        <Download className="h-3 w-3 mr-1" />
        Download
      </Button>
    );
    if (ent.quotaRemaining > 0) return (
      <Button onClick={handleDownload} size="sm" className="text-xs px-2 py-1 h-7" data-testid={`download-credit-${asset.id}`}>
        <Download className="h-3 w-3 mr-1" />
        Download (uses 1 credit)
      </Button>
    );
    return (
      <Button onClick={handlePurchase} size="sm" className="text-xs px-2 py-1 h-7" data-testid={`purchase-${asset.id}`}>
        <ShoppingCart className="h-3 w-3 mr-1" />
        Purchase $6.99
      </Button>
    );
  })();

  return (
    <Card className="group overflow-hidden hover:shadow-lg transition-all duration-300">
      <div 
        className="aspect-video bg-gray-100 dark:bg-gray-800 overflow-hidden relative cursor-pointer"
        onClick={() => onImageClick(asset)}
      >
        <img 
          src={asset.previewUrl} 
          alt={asset.name} 
          className="w-full h-full object-cover group-hover:scale-105 transition-transform duration-300" 
          loading="lazy"
        />
        <div className="absolute inset-0 bg-black/0 group-hover:bg-black/20 transition-colors duration-300 flex items-center justify-center opacity-0 group-hover:opacity-100">
          <div className="bg-white/90 hover:bg-white text-gray-900 p-2 rounded-lg transition-colors duration-200">
            <ZoomIn className="h-5 w-5" />
          </div>
        </div>
      </div>
      <CardContent className="p-4">
        <div className="mb-2">
          <h3 className="font-semibold text-sm mb-1 line-clamp-1" data-testid={`title-${asset.id}`}>
            {displayName}
          </h3>
          <p className="text-xs text-gray-500">File type: {type || "—"}</p>
        </div>
        <div className="flex items-center justify-between mb-2">
          <div className="flex flex-wrap gap-1">
            {asset.category && (
              <Badge variant="secondary" className="text-xs">
                {asset.category}
              </Badge>
            )}
          </div>
          {button}
        </div>
        <div className="text-xs text-gray-500 truncate mb-2">{asset.name}</div>
        {asset.tags && (
          <div className="flex flex-wrap gap-1">
            {asset.tags.slice(0, 3).map((tag, index) => (
              <Badge key={index} variant="outline" className="text-xs">
                {tag}
              </Badge>
            ))}
            {asset.tags.length > 3 && (
              <Badge variant="outline" className="text-xs">
                +{asset.tags.length - 3}
              </Badge>
            )}
          </div>
        )}
      </CardContent>
    </Card>
  );
}