import React, { useEffect, useState } from "react";
import { ensureFirebase, isFirebaseConfigured, signInWithGoogle, startMagicLink, doSignOut } from "../lib/firebase";
import { LogIn, LogOut, Mail, User } from "lucide-react";

export default function SignInChip() {
  const [uid, setUid] = useState<string | null>(null);
  const [email, setEmail] = useState("");
  const [sending, setSending] = useState(false);
  const [open, setOpen] = useState(false);
  const enabled = isFirebaseConfigured();

  useEffect(() => {
    const fb = ensureFirebase();
    if (!fb) return;
    fb.authReady?.then((user) => setUid(user?.uid ?? null));
    // reflect future auth changes
    if (fb) {
      fb.auth.onAuthStateChanged((u) => setUid(u?.uid ?? null));
    }
  }, []);

  if (!enabled) return null; // no chip if firebase not configured

  return (
    <div className="relative">
      <button
        onClick={() => setOpen((s) => !s)}
        className="inline-flex items-center gap-2 px-3 py-1.5 rounded-full border border-orange-500/30 bg-orange-500 text-white text-sm hover:bg-orange-600"
        title={uid ? `Signed in (${uid.slice(0,6)}…)` : "Sign in"}
        data-testid="button-signin-chip"
      >
        <User className="w-4 h-4" />
        {uid ? "Account" : "Sign in"}
      </button>

      {open && (
        <div className="absolute right-0 mt-2 w-64 rounded-xl border border-white/10 bg-slate-900/95 p-3 z-50" data-testid="dropdown-signin">
          {!uid ? (
            <div className="space-y-3">
              <button
                onClick={async () => { await signInWithGoogle(); setOpen(false); }}
                className="w-full inline-flex items-center justify-center gap-2 px-3 py-2 rounded-lg bg-white text-slate-900 hover:bg-slate-200"
                data-testid="button-google-signin"
              >
                <LogIn className="w-4 h-4" /> Continue with Google
              </button>

              {import.meta.env.VITE_FIREBASE_ENABLE_EMAIL_LINK && (
                <div className="space-y-2">
                  <div className="text-xs text-slate-400">or use Magic Link</div>
                  <div className="flex gap-2">
                    <input
                      type="email"
                      value={email}
                      onChange={(e) => setEmail(e.target.value)}
                      placeholder="you@example.com"
                      className="flex-1 rounded-lg bg-slate-950/60 border border-white/10 p-2 text-slate-100 placeholder:text-slate-500"
                      data-testid="input-magic-link-email"
                    />
                    <button
                      onClick={async () => { if (!email) return; setSending(true); await startMagicLink(email).catch(()=>{}); setSending(false); setOpen(false); alert("Magic link sent! Check your email."); }}
                      disabled={sending || !email}
                      className="inline-flex items-center gap-2 px-3 py-2 rounded-lg border border-white/10 hover:bg-white/5 text-white"
                      data-testid="button-send-magic-link"
                    >
                      <Mail className="w-4 h-4" /> {sending ? "Sending…" : "Send"}
                    </button>
                  </div>
                </div>
              )}
            </div>
          ) : (
            <div className="space-y-2">
              <div className="text-xs text-slate-400">UID</div>
              <div className="text-sm font-mono break-all" data-testid="text-user-uid">{uid}</div>
              <button
                onClick={async () => { await doSignOut(); setOpen(false); }}
                className="w-full inline-flex items-center justify-center gap-2 px-3 py-2 rounded-lg border border-white/10 hover:bg-white/5 text-white mt-2"
                data-testid="button-signout"
              >
                <LogOut className="w-4 h-4" /> Sign out
              </button>
            </div>
          )}
        </div>
      )}
    </div>
  );
}