import React, { useState, useEffect } from 'react';
import { useLocation, Link } from 'wouter';
import { useAuth } from '@/contexts/AuthContext';
import { featureFlags } from '@/lib/featureFlags';
import {
  BarChart3,
  Lightbulb,
  MessageSquare,
  FileText,
  Share2,
  Globe,
  Link as LinkIcon,
  Server,
  LogOut,
  Briefcase,
  ChevronDown,
  ChevronRight,
  Layers,
  Presentation,
  GalleryVerticalEnd,
  CaseSensitive,
  Image,
  Boxes,
  Images,
  PieChart,
  Library,
  ShoppingBag,
  Store,
  User,
  Package,
  Wand2,
  Palette
} from 'lucide-react';
import { useLogo } from '@/hooks/useLogo';
import { bulletCubeBlack, bulletCubeWhite, logoComposerIcon, aiLogosIcon } from '@/assets';

const dashboardItem = {
  id: 'dashboard',
  name: 'Dashboard',
  icon: BarChart3,
  path: '/dashboard',
  enabled: true
};

const businessDevelopmentItems = [
  { id: 'business-name', name: 'Brand Name Wizard', icon: () => <Wand2 className="mr-3 h-4 w-4 transform scale-x-[-1]" />, path: '/business-development/brand-name-wizard', enabled: true },
  { id: 'slogan-generator', name: 'Slogan Generator', icon: MessageSquare, path: '/business-development/slogan-generator', enabled: featureFlags.sloganGenerator },
  { 
    id: 'logo-templates', 
    name: 'AI Logo Creator', 
    icon: () => <img src={aiLogosIcon} alt="AI Logo Creator" className="w-5 h-5 mr-2" />, 
    path: '/business-development/ai-logo-creator', 
    enabled: true 
  },
  {
    id: 'brand-kit',
    name: 'Brand Kit',
    icon: (isActive: any = false) => {
      const actualIsActive = typeof isActive === 'object' ? isActive?.isActive : isActive;
      return <img src={actualIsActive ? bulletCubeWhite : bulletCubeBlack} alt="Brand Kit" className="w-5 h-5 relative -ml-0.5 mr-3" />;
    },
    path: '/business-development/brand-kit',
    enabled: true
  },
  { id: 'business-plan', name: 'AI Business Plan Developer', icon: Boxes, path: '/business-development/ai-business-plan-developer', enabled: featureFlags.businessPlan },
  // Phase 2 placeholders
  { id: 'website-templates', name: 'Website Templates', icon: Globe, path: '/website-templates', enabled: false },
  { id: 'social-media', name: 'Social Media Kits', icon: Share2, path: '/social-kits', enabled: false },
];

const webServicesItems = [
  { id: 'domains', name: 'Find a Domain', icon: LinkIcon, path: '/web-services/domains', enabled: true },
  { id: 'hosting', name: 'Web Hosting', icon: Server, path: '/web-services/web-hosting', enabled: true },
];

const businessAssetsItems = [
  { id: 'business-plan-template', name: 'Business Plan Template', icon: FileText, path: '/business-assets/templates/business-plan', enabled: true },
  { 
    id: 'logo-templates', 
    name: 'Logo Templates', 
    icon: () => <img src={logoComposerIcon} alt="Logo Templates" className="w-5 h-5 mr-2" />, 
    path: '/business-assets/logo-templates', 
    enabled: true 
  },
];


export function Sidebar() {
  const [location, setLocation] = useLocation();
  const { currentUser, logout } = useAuth();
  const [backendUser, setBackendUser] = useState<any>(null);
  const [isBusinessDevOpen, setIsBusinessDevOpen] = useState(true);
  const [isWebServicesOpen, setIsWebServicesOpen] = useState(true);
  const [isBusinessAssetsOpen, setIsBusinessAssetsOpen] = useState(true);
  const [isMarketplaceOpen, setIsMarketplaceOpen] = useState(true);
  const logoPath = useLogo();

  // role helpers
  const isAdminLike = (u: any) => {
    const r = (u?.role || u?.roles || []).toString().toLowerCase();
    return r.includes('admin') || r.includes('owner') || r.includes('manager');
  };
  const isCreator = (u: any) => {
    const r = (u?.role || u?.roles || []).toString().toLowerCase();
    // Include explicit owner check since you're the business owner
    return r.includes('creator') || isAdminLike(u) || u?.isOwner;
  };

  // Fetch backend user (who am I + roles)
  useEffect(() => {
    const fetchBackendUser = async () => {
      if (!currentUser) { setBackendUser(null); return; }
      try {
        const token = await currentUser.getIdToken();
        const res = await fetch('/api/auth/me', { headers: { Authorization: `Bearer ${token}` } });
        if (res.ok) setBackendUser(await res.json());
      } catch (e) {
        console.error('Failed to fetch backend user profile:', e);
      }
    };
    fetchBackendUser();
  }, [currentUser]);

  const handleLogout = async () => {
    try { await logout(); } catch (e) { console.error('Failed to log out:', e); }
  };

  const getInitials = (name: string) => name.split(' ').map(n => n[0]).join('').toUpperCase();

  return (
    <div className="w-[18.5rem] flex-shrink-0 bg-card border-r border-border shadow-lg">
      <div className="flex flex-col h-full overflow-y-auto">
        {/* Logo */}
        <div className="flex items-center px-6 py-4 border-b border-border">
          <Link href="/">
            <div className="flex items-center cursor-pointer" data-testid="link-logo-sidebar">
              <img src={logoPath} alt="IBrandBiz" className="h-[4.2rem] w-auto" />
            </div>
          </Link>
        </div>

        {/* Navigation */}
        <nav className="flex-1 px-4 py-6 space-y-2">
          {/* Dashboard */}
          {dashboardItem.enabled && (
            <button
              key={dashboardItem.id}
              onClick={() => setLocation(dashboardItem.path)}
              className="w-full flex items-center px-3 py-2 rounded-lg text-lg font-bold transition-colors bg-[#ff8800] text-white"
              data-testid={`nav-${dashboardItem.id}`}
            >
              <BarChart3 className="mr-3 h-4 w-4" />
              {dashboardItem.name}
            </button>
          )}

          {/* Business Development */}
          <div className="space-y-1">
            <button
              onClick={() => setIsBusinessDevOpen(!isBusinessDevOpen)}
              className="w-full flex items-center px-3 py-2 rounded-lg text-sm font-medium text-muted-foreground hover:bg-lightgray hover:text-foreground transition-colors"
              data-testid="nav-business-development"
            >
              <Briefcase className="mr-3 h-4 w-4" />
              Business Development
              {isBusinessDevOpen ? <ChevronDown className="ml-auto h-4 w-4" /> : <ChevronRight className="ml-auto h-4 w-4" />}
            </button>

            {isBusinessDevOpen && (
              <div className="ml-6 space-y-1">
                {businessDevelopmentItems.filter(i => i.enabled).map((item) => {
                  const Icon = item.icon as any;
                  const isActive = location === item.path;
                  return (
                    <button
                      key={item.id}
                      onClick={() => setLocation(item.path)}
                      className={`w-full flex items-center px-3 py-2 rounded-lg text-sm transition-colors ${
                        isActive ? 'bg-primary text-white font-bold' : 'text-muted-foreground hover:bg-lightgray hover:text-foreground font-medium'
                      }`}
                      data-testid={`nav-${item.id}`}
                    >
                      {item.id === 'brand-kit' ? <Icon isActive={isActive} /> : (item.id === 'business-name' ? <Icon /> : <Icon className="mr-3 h-4 w-4" />)}
                      {item.name}
                    </button>
                  );
                })}
              </div>
            )}
          </div>


          {/* Business Assets */}
          <div className="space-y-1">
            <button
              onClick={() => setIsBusinessAssetsOpen(!isBusinessAssetsOpen)}
              className="w-full flex items-center px-3 py-2 rounded-lg text-sm font-medium text-muted-foreground hover:bg-lightgray hover:text-foreground transition-colors"
              data-testid="nav-business-assets"
            >
              <Layers className="mr-3 h-4 w-4" />
              Business Assets
              {isBusinessAssetsOpen ? <ChevronDown className="ml-auto h-4 w-4" /> : <ChevronRight className="ml-auto h-4 w-4" />}
            </button>

            {isBusinessAssetsOpen && (
              <div className="ml-6 space-y-1">
                {businessAssetsItems.filter(i => i.enabled).map((item) => {
                  const Icon = item.icon as any;
                  const isActive = location === item.path;
                  return (
                    <button
                      key={item.id}
                      onClick={() => setLocation(item.path)}
                      className={`w-full flex items-center px-3 py-2 rounded-lg text-sm transition-colors ${
                        isActive ? 'bg-primary text-white font-bold' : 'text-muted-foreground hover:bg-lightgray hover:text-foreground font-medium'
                      }`}
                      data-testid={`nav-${item.id}`}
                    >
                      {item.id === 'logo-templates' ? <Icon /> : <Icon className="mr-3 h-4 w-4" />}
                      {item.name}
                    </button>
                  );
                })}
              </div>
            )}
          </div>


          {/* Web Services */}
          <div className="space-y-1">
            <button
              onClick={() => setIsWebServicesOpen(!isWebServicesOpen)}
              className="w-full flex items-center px-3 py-2 rounded-lg text-sm font-medium text-muted-foreground hover:bg-lightgray hover:text-foreground transition-colors"
              data-testid="nav-web-services"
            >
              <Globe className="mr-3 h-4 w-4" />
              Web Services
              {isWebServicesOpen ? <ChevronDown className="ml-auto h-4 w-4" /> : <ChevronRight className="ml-auto h-4 w-4" />}
            </button>

            {isWebServicesOpen && (
              <div className="ml-6 space-y-1">
                {webServicesItems.filter(i => i.enabled).map((item) => {
                  const Icon = item.icon;
                  const isActive = location === item.path;
                  return (
                    <button
                      key={item.id}
                      onClick={() => setLocation(item.path)}
                      className={`w-full flex items-center px-3 py-2 rounded-lg text-sm transition-colors ${
                        isActive ? 'bg-primary text-white font-bold' : 'text-muted-foreground hover:bg-lightgray hover:text-foreground font-medium'
                      }`}
                      data-testid={`nav-${item.id}`}
                    >
                      <Icon className="mr-3 h-4 w-4" />
                      {item.name}
                    </button>
                  );
                })}
              </div>
            )}
          </div>

        </nav>

        {/* User Menu */}
        <div className="px-4 py-4 border-t border-border">
          {/* Creator Marketplace - creator-only */}
          {(isCreator(backendUser) || backendUser?.isOwner) && (
            <div className="mb-3">
              <button
                onClick={() => setLocation('/creator/my-marketplace')}
                className={`w-full flex items-center px-3 py-2 rounded-lg text-sm font-medium transition-colors ${
                  location.startsWith('/creator/my-marketplace')
                    ? 'bg-primary text-white font-bold'
                    : 'text-muted-foreground hover:bg-lightgray hover:text-foreground'
                }`}
                data-testid="nav-my-marketplace"
              >
                <ShoppingBag className="mr-3 h-4 w-4" />
                My Marketplace
              </button>
            </div>
          )}
          
          <div className="mb-3">
            <button
              onClick={() => setLocation('/account/purchases')}
              className="w-full flex items-center px-3 py-2 rounded-lg text-sm font-medium transition-colors text-muted-foreground hover:bg-lightgray hover:text-foreground"
              data-testid="nav-my-purchases"
            >
              <Package className="mr-3 h-4 w-4" />
              My Purchases
            </button>
          </div>
          
          <div className="flex items-center">
            <button
              onClick={() => setLocation('/profile')}
              className="flex-1 flex items-center p-2 rounded-lg hover:bg-gray-50 dark:hover:bg-gray-800 transition-colors"
              data-testid="button-profile"
            >
              <div className="w-8 h-8 bg-accent rounded-full flex items-center justify-center mr-3">
                <span className="text-accent-foreground text-sm font-medium">
                  {(backendUser?.displayName || currentUser?.displayName) ? getInitials(backendUser?.displayName || currentUser?.displayName) : 'U'}
                </span>
              </div>
              <div className="flex-1 text-left">
                <p className="text-sm font-medium text-foreground" data-testid="text-username">
                  {backendUser?.displayName || currentUser?.displayName || 'User'}
                </p>
                <p className="text-xs text-muted-foreground" data-testid="text-email">
                  {backendUser?.email || currentUser?.email}
                </p>
              </div>
            </button>
            <button
              onClick={handleLogout}
              className="text-muted-foreground hover:text-foreground p-2 rounded-lg hover:bg-gray-50 dark:hover:bg-gray-800 transition-colors"
              data-testid="button-logout"
              title="Log out"
            >
              <LogOut className="h-4 w-4" />
            </button>
          </div>
        </div>
      </div>
    </div>
  );
}

