import { useState } from "react";
import { Search } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { useLocation } from "wouter";

export default function SearchBar() {
  const [searchQuery, setSearchQuery] = useState("");
  const [, setLocation] = useLocation();

  const handleSearch = (e: React.FormEvent) => {
    e.preventDefault();
    if (!searchQuery.trim()) return;

    const query = searchQuery.trim().toLowerCase();
    
    // Smart routing based on search query
    if (query.includes("domain") || query.includes(".com") || query.includes(".net") || query.includes(".org")) {
      setLocation(`/domains?search=${encodeURIComponent(query)}`);
    } else if (query.includes("template") || query.includes("website")) {
      setLocation("/website-templates");
    } else if (query.includes("social") || query.includes("kit")) {
      setLocation("/social-kits");
    } else if (query.includes("brand") || query.includes("logo")) {
      setLocation("/brand-kit");
    } else if (query.includes("name") || query.includes("business")) {
      setLocation("/name-wizard");
    } else {
      // Default to domains search for generic queries
      setLocation(`/domains?search=${encodeURIComponent(query)}`);
    }
  };

  return (
    <div className="w-full max-w-4xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
      <form onSubmit={handleSearch} className="relative">
        <div className="relative flex items-center">
          <Search 
            className="absolute left-4 h-5 w-5 text-muted-foreground pointer-events-none"
            data-testid="icon-search"
          />
          <Input
            type="text"
            placeholder="Search domains, brand kits, or templates…"
            value={searchQuery}
            onChange={(e) => setSearchQuery(e.target.value)}
            className="pl-12 pr-24 py-6 text-base bg-white dark:bg-gray-800 border-2 border-gray-200 dark:border-gray-600 focus:border-primary dark:focus:border-primary rounded-full shadow-lg focus:shadow-xl transition-all duration-300"
            data-testid="input-global-search"
          />
          <Button
            type="submit"
            className="absolute right-2 bg-primary hover:bg-primary/90 text-white rounded-full px-6 py-2 font-semibold"
            data-testid="button-search-submit"
          >
            Search
          </Button>
        </div>
      </form>
      
      {/* Quick search suggestions */}
      <div className="flex flex-wrap gap-2 mt-4 justify-center items-center">
        <span className="text-sm text-muted-foreground flex items-center h-8">Popular searches:</span>
        <button
          onClick={() => setLocation("/name-wizard")}
          className="text-sm bg-gray-100 dark:bg-gray-700 hover:bg-gray-200 dark:hover:bg-gray-600 text-gray-700 dark:text-gray-300 px-3 py-1 rounded-full transition-colors"
          data-testid="button-quick-search-business-names"
        >
          business names
        </button>
        <button
          onClick={() => setLocation("/domains")}
          className="text-sm bg-gray-100 dark:bg-gray-700 hover:bg-gray-200 dark:hover:bg-gray-600 text-gray-700 dark:text-gray-300 px-3 py-1 rounded-full transition-colors"
          data-testid="button-quick-search-domains"
        >
          domains
        </button>
        <button
          onClick={() => setLocation("/website-templates")}
          className="text-sm bg-gray-100 dark:bg-gray-700 hover:bg-gray-200 dark:hover:bg-gray-600 text-gray-700 dark:text-gray-300 px-3 py-1 rounded-full transition-colors"
          data-testid="button-quick-search-templates"
        >
          website templates
        </button>
        <button
          onClick={() => setLocation("/brand-kit")}
          className="text-sm bg-gray-100 dark:bg-gray-700 hover:bg-gray-200 dark:hover:bg-gray-600 text-gray-700 dark:text-gray-300 px-3 py-1 rounded-full transition-colors"
          data-testid="button-quick-search-branding"
        >
          branding
        </button>
      </div>
    </div>
  );
}