import { useState } from "react";
import { Button } from "@/components/ui/button";
import { Textarea } from "@/components/ui/textarea";
import { Download, Loader2 } from "lucide-react";
import { apiRequest } from "@/lib/queryClient";

type GenState = "idle" | "running" | "done" | "error";

interface RecraftGeneratorProps {
  businessName?: string;
  className?: string;
  onComplete?: (svgUrl: string) => void;
}

export default function RecraftGenerator({ businessName = "", className = "", onComplete }: RecraftGeneratorProps) {
  const [prompt, setPrompt] = useState("");
  const [status, setStatus] = useState<GenState>("idle");
  const [error, setError] = useState<string | null>(null);
  const [svgUrl, setSvgUrl] = useState<string | null>(null);
  const [predictionId, setPredictionId] = useState<string | null>(null);

  async function onGenerate() {
    if (!prompt.trim()) {
      setError("Please enter a prompt description");
      return;
    }

    setStatus("running");
    setError(null);
    setSvgUrl(null);
    setPredictionId(null);

    try {
      // Step 1: Create prediction using authenticated API request
      const createJson = await apiRequest("POST", "/api/recraft/create", {
        prompt: prompt.trim()
      });

      const { predictionId, version } = createJson;
      setPredictionId(predictionId);
      console.log(`[recraft] created prediction ${predictionId} with version ${version}`);

      // Step 2: Poll for status
      await pollForResult(predictionId);
    } catch (e: any) {
      setError(e.message || "Something went wrong");
      setStatus("error");
    }
  }

  async function pollForResult(predictionId: string) {
    const timeoutAt = Date.now() + 210_000; // 3.5 minutes
    let pollCount = 0;

    while (Date.now() < timeoutAt) {
      pollCount++;
      
      try {
        const statusJson = await apiRequest("GET", `/api/recraft/status/${predictionId}`);

        console.log(`[recraft] poll #${pollCount} for ${predictionId}: ${statusJson.status}`);

        if (statusJson.status === "succeeded") {
          if (statusJson.svgUrl) {
            // Use direct URL - blob processing was causing display issues
            console.log("[recraft] using direct SVG URL:", statusJson.svgUrl);
            setSvgUrl(statusJson.svgUrl);
            setStatus("done");
            console.log("[recraft] success! SVG ready for display");
            
            // Call the completion callback with direct URL
            if (onComplete) {
              onComplete(statusJson.svgUrl);
            }
            return;
          } else {
            // Debug info for failed URL extraction
            console.error("[recraft] succeeded but no SVG URL:", statusJson);
            const debugMsg = statusJson.debugOutput 
              ? `No valid SVG URL found. Output shape: ${statusJson.rawOutputShape}. Debug: ${JSON.stringify(statusJson.debugOutput).substring(0, 200)}...`
              : `No valid SVG URL found. Output shape: ${statusJson.rawOutputShape}`;
            throw new Error(debugMsg);
          }
        }

        if (statusJson.status === "failed") {
          throw new Error(statusJson.error || "Generation failed on server");
        }

        // Still processing (queued, starting, processing) - continue polling
        if (statusJson.status === "queued") {
          // Poll less frequently when queued to reduce server load
          await new Promise(resolve => setTimeout(resolve, 3000)); // 3s for queued
        } else {
          // Poll more frequently when processing
          await new Promise(resolve => setTimeout(resolve, 2000)); // 2s for processing
        }
        
      } catch (e: any) {
        // If it's a polling error, throw it to stop polling
        if (e.message.includes("fetch") || e.message.includes("Network")) {
          throw new Error(`Network error during polling: ${e.message}`);
        }
        throw e;
      }
    }

    // Timeout reached
    throw new Error(`Generation timed out after ${pollCount} polls (3.5 minutes). The request may still be processing on the server.`);
  }

  async function onDownload() {
    if (!svgUrl) return;
    
    try {
      const resp = await fetch(svgUrl);
      const svgText = await resp.text();
      
      // Download the original SVG (transparency processing was causing issues)
      const blob = new Blob([svgText], { type: "image/svg+xml;charset=utf-8" });
      const url = URL.createObjectURL(blob);

      const a = document.createElement("a");
      a.href = url;
      a.download = makeFileName(prompt);
      a.click();

      URL.revokeObjectURL(url);
    } catch (e) {
      setError("Failed to download SVG");
    }
  }


  return (
    <div className={`w-full max-w-4xl space-y-6 ${className}`}>
      <div className="space-y-3">
        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300">
          Describe your logo
        </label>
        <Textarea
          className="w-full min-h-[100px] resize-none"
          value={prompt}
          onChange={(e) => setPrompt(e.target.value)}
          placeholder={businessName 
            ? `Describe your ${businessName} logo (e.g., minimal ${businessName} icon, geometric, vector, no text)` 
            : "Describe your logo (e.g., minimal coffee bean icon, geometric, vector, no text)"
          }
          disabled={status === "running"}
          data-testid="textarea-logo-prompt"
        />
      </div>

      <div className="flex items-center gap-3 flex-wrap">
        <Button
          onClick={onGenerate}
          disabled={status === "running" || !prompt.trim()}
          className="bg-emerald-600 hover:bg-emerald-700 text-white px-6 py-2 min-w-[140px]"
          data-testid="button-generate-logo"
        >
          {status === "running" ? (
            <>
              <Loader2 className="w-4 h-4 mr-2 animate-spin" />
              Generating...
            </>
          ) : (
            "Generate Logo"
          )}
        </Button>

        {svgUrl && (
          <Button
            onClick={onDownload}
            variant="outline"
            className="border-emerald-600 text-emerald-600 hover:bg-emerald-50 px-4 py-2"
            data-testid="button-download-logo"
          >
            <Download className="w-4 h-4 mr-2" />
            Download SVG
          </Button>
        )}

        {status === "running" && (
          <span className="text-sm text-gray-600 dark:text-gray-400">
            This may take up to 3 minutes...
          </span>
        )}
      </div>


      {error && (
        <div className="rounded-lg border border-red-300 bg-red-50 dark:bg-red-900/20 dark:border-red-800 p-4 text-sm text-red-700 dark:text-red-400" data-testid="error-message">
          <strong>Error:</strong> {error}
        </div>
      )}

      <div className="rounded-xl border border-gray-200 dark:border-gray-700 bg-white dark:bg-gray-900 p-6">
        <div className="mb-3 text-sm font-medium text-gray-700 dark:text-gray-300">Preview</div>
        <div
          className="flex items-center justify-center rounded-lg border-2 border-dashed border-gray-300 dark:border-gray-600 relative"
          style={{ 
            minHeight: "280px",
            backgroundImage: 'url("data:image/svg+xml,%3csvg width=\'20\' height=\'20\' xmlns=\'http://www.w3.org/2000/svg\'%3e%3cdefs%3e%3cpattern id=\'a\' patternUnits=\'userSpaceOnUse\' width=\'20\' height=\'20\'%3e%3crect x=\'0\' y=\'0\' width=\'10\' height=\'10\' fill=\'%23f3f4f6\'/%3e%3crect x=\'10\' y=\'10\' width=\'10\' height=\'10\' fill=\'%23f3f4f6\'/%3e%3c/pattern%3e%3c/defs%3e%3crect width=\'100%25\' height=\'100%25\' fill=\'url(%23a)\'/%3e%3c/svg%3e")',
            backgroundSize: '20px 20px'
          }}
          data-testid="logo-preview-container"
        >
          {!svgUrl && status !== "running" && (
            <div className="text-center text-gray-500 dark:text-gray-400 bg-white/90 dark:bg-gray-900/90 p-4 rounded-lg">
              <p className="text-sm">No logo generated yet</p>
              <p className="text-xs mt-1">Click "Generate Logo" to create a professional SVG logo with transparent background</p>
            </div>
          )}
          {status === "running" && (
            <div className="text-center text-gray-500 dark:text-gray-400 bg-white/90 dark:bg-gray-900/90 p-4 rounded-lg">
              <Loader2 className="w-8 h-8 mx-auto mb-2 animate-spin" />
              <p className="text-sm">Generating your transparent logo...</p>
              <p className="text-xs mt-1">This may take up to 3 minutes</p>
            </div>
          )}
          {svgUrl && (
            <div className="flex flex-col items-center gap-2">
              <img
                src={svgUrl}
                alt="Generated SVG logo with transparent background"
                className="max-w-full max-h-full object-contain"
                style={{ maxHeight: "240px", maxWidth: "240px" }}
                data-testid="generated-logo-image"
              />
            </div>
          )}
        </div>
      </div>
    </div>
  );
}

function makeFileName(prompt: string): string {
  const base = prompt
    .toLowerCase()
    .replace(/[^a-z0-9]+/g, "-")
    .replace(/(^-|-$)/g, "") || "logo";
  return `${base}.svg`;
}