import { useEffect, useState } from 'react';
import { useLocation } from 'wouter';
import { useAuth } from '@/contexts/AuthContext';
import { AuthModal } from '@/components/AuthModal';

interface ProtectedRouteProps {
  children: React.ReactNode;
}

export function ProtectedRoute({ children }: ProtectedRouteProps) {
  const { currentUser, loading } = useAuth();
  const [isModalOpen, setIsModalOpen] = useState(false);
  const [, navigate] = useLocation();

  // Auto-open modal when user is not authenticated (after loading completes)
  useEffect(() => {
    if (!loading) {
      setIsModalOpen(!currentUser);
    }
  }, [currentUser, loading]);

  // Handle modal close - only redirect to home if user is still not authenticated
  const handleClose = () => {
    setIsModalOpen(false);
    // Don't redirect if user is now authenticated (successful login)
    if (!currentUser) {
      navigate('/');
    }
  };

  // Show loading state while auth is initializing to prevent flicker
  if (loading) {
    return null;
  }

  return (
    <>
      {/* Show protected content, but blur it when modal is open */}
      <div className={isModalOpen && !currentUser ? "blur-sm pointer-events-none" : ""}>
        {children}
      </div>
      
      {/* Show auth modal when user is not authenticated */}
      <AuthModal isOpen={isModalOpen} onClose={handleClose} />
    </>
  );
}
