import React, { useState } from 'react';
import { useAuth } from '@/contexts/AuthContext';
import { useLocation } from 'wouter';
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogFooter,
  DialogHeader,
  DialogTitle,
} from "@/components/ui/dialog";
import { Button } from "@/components/ui/button";
import { Crown, Check, Loader2, X } from "lucide-react";
import { apiRequest } from '@/lib/queryClient';
import { useToast } from '@/hooks/use-toast';

interface ProUpgradeModalProps {
  isOpen: boolean;
  onClose: () => void;
  billing?: 'monthly' | 'yearly';
}

export function ProUpgradeModal({ 
  isOpen, 
  onClose, 
  billing = 'monthly' 
}: ProUpgradeModalProps) {
  const { currentUser } = useAuth();
  const [, navigate] = useLocation();
  const { toast } = useToast();
  const [loading, setLoading] = useState(false);

  const pricing = {
    monthly: { amount: '$19', period: 'per month', total: '$19/month', savings: '' },
    yearly: { amount: '$190', period: 'per year', total: '$15.83/month', savings: 'Save $38' }
  };

  const currentPlan = pricing[billing];

  const features = [
    'Unlimited Business Name generations',
    'Complete Brand Kit creation',
    'Slogan Generator with AI suggestions', 
    'Business Plan Developer',
    'Domain search integration',
    'Export to PDF, DOCX, Google Docs',
    'Premium logo composers',
    'Priority support',
    'Commercial usage rights'
  ];

  const handleUpgrade = async () => {
    if (!currentUser) {
      toast({
        title: "Authentication Required",
        description: "Please sign in to upgrade to Pro.",
        variant: "destructive"
      });
      return;
    }

    try {
      setLoading(true);
      
      // First, ensure user record exists in database
      const token = await currentUser.getIdToken();
      
      // Check if user exists, create if not
      let userExists = false;
      try {
        const checkUserResponse = await fetch('/api/auth/me', {
          headers: {
            'Authorization': `Bearer ${token}`,
            'Content-Type': 'application/json'
          }
        });
        userExists = checkUserResponse.ok;
      } catch (error) {
        console.log('User check failed, will create user record');
      }

      if (!userExists) {
        // User doesn't exist in database, create them
        console.log('Creating user record...');
        await apiRequest('POST', '/api/users', {
          firebaseUid: currentUser.uid,
          email: currentUser.email || '',
          displayName: currentUser.displayName || '',
          role: 'user'
        });
        
        // Wait a moment for user creation to complete
        await new Promise(resolve => setTimeout(resolve, 500));
      }

      // Now proceed with checkout
      console.log('Proceeding with checkout...');
      const response = await apiRequest('POST', '/api/billing/checkout', {
        priceType: billing,
        successUrl: `${window.location.origin}/dashboard?welcome=pro`,
        cancelUrl: `${window.location.origin}/pricing?cancelled=true`
      });
      
      const data = await response.json();
      const { checkoutUrl } = data;
      
      if (checkoutUrl) {
        // Redirect to Stripe Checkout
        window.location.href = checkoutUrl;
      } else {
        throw new Error('No checkout URL received');
      }
    } catch (error: any) {
      console.error('Checkout error:', error);
      
      let errorMessage = "Unable to set up payment. Please try again.";
      
      if (error.message?.includes('401')) {
        errorMessage = "Please sign in to continue with your Pro subscription.";
      } else if (error.message?.includes('404')) {
        errorMessage = "Account setup failed. Please try refreshing the page and trying again.";
      } else if (error.message?.includes('500')) {
        errorMessage = "Payment system temporarily unavailable. Please try again later.";
      } else if (error.message?.includes('configuration')) {
        errorMessage = "Payment setup incomplete. Contact support for assistance.";
      }
      
      toast({
        title: "Payment Setup Failed",
        description: errorMessage,
        variant: "destructive"
      });
    } finally {
      setLoading(false);
    }
  };

  return (
    <Dialog open={isOpen} onOpenChange={onClose}>
      <DialogContent 
        className="sm:max-w-[500px] p-0 overflow-hidden" 
        data-testid="pro-upgrade-modal"
      >
        {/* Header with gradient background */}
        <div className="bg-gradient-to-r from-green-500 to-teal-500 p-6 text-white relative">
          <button 
            onClick={onClose}
            className="absolute top-4 right-4 text-white/80 hover:text-white transition-colors"
            data-testid="button-close-modal"
          >
            <X className="h-5 w-5" />
          </button>
          
          <DialogHeader className="text-white">
            <div className="flex items-center gap-3 mb-2">
              <Crown className="h-8 w-8 text-yellow-300" />
              <DialogTitle className="text-2xl font-bold text-white" data-testid="modal-title">
                Upgrade to Pro
              </DialogTitle>
            </div>
            <DialogDescription className="text-white/90 text-base" data-testid="modal-description">
              Unlock all premium features and take your brand to the next level
            </DialogDescription>
          </DialogHeader>
        </div>

        {/* Content */}
        <div className="p-6">
          {/* Pricing section */}
          <div className="text-center mb-6">
            <div className="flex items-baseline justify-center gap-2 mb-2">
              <span className="text-4xl font-bold text-green-600" data-testid="price-amount">
                {currentPlan.amount}
              </span>
              <span className="text-gray-600" data-testid="price-period">
                {currentPlan.period}
              </span>
            </div>
            {billing === 'yearly' && (
              <div className="inline-flex items-center gap-2 px-3 py-1 bg-green-100 text-green-700 rounded-full text-sm font-medium">
                <Check className="h-4 w-4" />
                <span data-testid="savings-badge">{currentPlan.savings}</span>
              </div>
            )}
            <p className="text-sm text-gray-600 mt-2" data-testid="price-calculation">
              {billing === 'yearly' ? 'Billed annually • ' + currentPlan.total : 'Billed monthly'}
            </p>
          </div>

          {/* Features */}
          <div className="space-y-3 mb-6">
            <h4 className="font-semibold text-gray-900 mb-3">What's included:</h4>
            <div className="grid grid-cols-1 gap-2 max-h-48 overflow-y-auto">
              {features.map((feature, index) => (
                <div key={index} className="flex items-start gap-3" data-testid={`feature-${index}`}>
                  <Check className="h-4 w-4 text-green-500 mt-0.5 flex-shrink-0" />
                  <span className="text-sm text-gray-700">{feature}</span>
                </div>
              ))}
            </div>
          </div>

          {/* Action buttons */}
          <div className="space-y-3">
            <Button 
              onClick={handleUpgrade}
              disabled={loading}
              className="w-full bg-green-600 hover:bg-green-700 text-white py-3 text-lg font-semibold"
              data-testid="button-upgrade-to-pro"
            >
              {loading ? (
                <>
                  <Loader2 className="mr-2 h-5 w-5 animate-spin" />
                  Setting up payment...
                </>
              ) : (
                `Upgrade to Pro - ${currentPlan.amount}`
              )}
            </Button>
            
            <Button 
              variant="outline" 
              onClick={onClose}
              className="w-full"
              disabled={loading}
              data-testid="button-cancel-upgrade"
            >
              Maybe Later
            </Button>
          </div>

          {/* Trust indicators */}
          <div className="mt-4 text-center">
            <p className="text-xs text-gray-500">
              💳 Secure payment • Cancel anytime • 30-day money-back guarantee
            </p>
          </div>
        </div>
      </DialogContent>
    </Dialog>
  );
}