import React, { createContext, useContext, useEffect, useState, useRef, useCallback } from 'react';
import { useAuth } from '@/contexts/AuthContext';
import { toast } from 'sonner';
import { useQuery, useQueryClient } from '@tanstack/react-query';
import { Bell, Check, AlertTriangle, Info, CheckCircle, X } from 'lucide-react';

interface Notification {
  id: string;
  userId: string;
  message: string;
  type: 'info' | 'success' | 'warning' | 'error' | 'pro_welcome' | 'system';
  isRead: boolean;
  createdAt: string;
}

interface NotificationContextType {
  notifications: Notification[];
  unreadCount: number;
  isConnected: boolean;
  isReconnecting: boolean;
  connectionError: string | null;
}

const NotificationContext = createContext<NotificationContextType>({
  notifications: [],
  unreadCount: 0,
  isConnected: false,
  isReconnecting: false,
  connectionError: null,
});

export const useNotifications = () => useContext(NotificationContext);

interface SSEMessage {
  type: 'connected' | 'notification' | 'keepalive' | 'error';
  message?: string;
  data?: Notification;
  timestamp?: number;
}

export function NotificationsProvider({ children }: { children: React.ReactNode }) {
  const { currentUser } = useAuth();
  const queryClient = useQueryClient();
  const [isConnected, setIsConnected] = useState(false);
  const [isReconnecting, setIsReconnecting] = useState(false);
  const [connectionError, setConnectionError] = useState<string | null>(null);
  const eventSourceRef = useRef<EventSource | null>(null);
  const reconnectTimeoutRef = useRef<NodeJS.Timeout>();
  const reconnectAttemptsRef = useRef(0);
  const maxReconnectAttempts = 5;
  const baseReconnectDelay = 2000;

  // Fetch initial notifications (non-streaming)
  const { data: notifications = [] } = useQuery<Notification[]>({
    queryKey: ['/api/notifications'],
    enabled: !!currentUser,
    staleTime: 0, // Always fetch fresh data since we have SSE updates
  });

  const unreadCount = notifications.filter(n => !n.isRead).length;

  const getToastIcon = useCallback((type: string) => {
    switch (type) {
      case 'success':
        return <CheckCircle className="h-4 w-4 text-green-600" />;
      case 'warning':
        return <AlertTriangle className="h-4 w-4 text-yellow-600" />;
      case 'error':
        return <X className="h-4 w-4 text-red-600" />;
      case 'pro_welcome':
        return <Check className="h-4 w-4 text-purple-600" />;
      default:
        return <Info className="h-4 w-4 text-blue-600" />;
    }
  }, []);

  const showNotificationToast = useCallback((notification: Notification) => {
    const icon = getToastIcon(notification.type);
    
    // Configure toast based on notification type
    switch (notification.type) {
      case 'success':
        toast.success(notification.message, {
          icon: icon,
          duration: 5000,
        });
        break;
      case 'error':
        toast.error(notification.message, {
          icon: icon,
          duration: 8000,
        });
        break;
      case 'warning':
        toast.warning(notification.message, {
          icon: icon,
          duration: 6000,
        });
        break;
      case 'pro_welcome':
        toast.success(notification.message, {
          icon: icon,
          duration: 10000,
          action: {
            label: 'Got it',
            onClick: () => {
              // Handle pro welcome dismissal if needed
              console.log('Pro welcome notification acknowledged');
            },
          },
        });
        break;
      default:
        toast(notification.message, {
          icon: icon,
          duration: 5000,
        });
        break;
    }
  }, [getToastIcon]);

  const establishSSEConnection = useCallback(async () => {
    if (!currentUser || eventSourceRef.current) return;

    try {
      setIsReconnecting(reconnectAttemptsRef.current > 0);
      setConnectionError(null);

      // SECURITY FIX: Get secure stream token instead of using Firebase token in URL
      let streamToken: string;
      try {
        // Get Firebase ID token for backend authentication
        const firebaseToken = await currentUser.getIdToken();
        
        // Request secure stream token from backend
        const response = await fetch('/api/notifications/stream-token', {
          method: 'POST',
          headers: {
            'Authorization': `Bearer ${firebaseToken}`,
            'Content-Type': 'application/json'
          }
        });

        if (!response.ok) {
          throw new Error(`Failed to get stream token: ${response.status}`);
        }

        const { streamToken: token } = await response.json();
        streamToken = token;
      } catch (error) {
        // Only log error in production, suppress in development
        if (import.meta.env.MODE !== 'development') {
          console.error('Failed to get secure stream token:', error);
        }
        
        // Development fallback for testing
        const isDevelopment = import.meta.env.MODE === 'development';
        if (isDevelopment) {
          try {
            const response = await fetch('/api/notifications/stream-token', {
              method: 'POST',
              headers: {
                'x-dev-bypass': 'development',
                'Content-Type': 'application/json'
              }
            });
            
            if (response.ok) {
              const { streamToken: token } = await response.json();
              streamToken = token;
            } else {
              throw new Error('Development fallback failed');
            }
          } catch (devError) {
            // Suppress development errors when database is unavailable
            console.warn('Notifications unavailable in development (database connection failed)');
            setConnectionError(null); // Don't show error in development
            setIsConnected(false);
            setIsReconnecting(false);
            return;
          }
        } else {
          setConnectionError('Authentication failed');
          return;
        }
      }
      
      // Create SSE connection with secure opaque token
      const eventSource = new EventSource(`/api/notifications/stream?token=${encodeURIComponent(streamToken)}`);

      eventSourceRef.current = eventSource;

      eventSource.onopen = () => {
        console.log('SSE connection established');
        setIsConnected(true);
        setIsReconnecting(false);
        setConnectionError(null);
        reconnectAttemptsRef.current = 0;
      };

      eventSource.onmessage = (event) => {
        try {
          const message: SSEMessage = JSON.parse(event.data);
          
          switch (message.type) {
            case 'connected':
              console.log('SSE connected:', message.message);
              break;
              
            case 'notification':
              if (message.data) {
                console.log('New notification received:', message.data);
                
                // Show toast notification
                showNotificationToast(message.data);
                
                // Update query cache with new notification
                queryClient.setQueryData(['/api/notifications'], (old: Notification[] = []) => {
                  return [message.data, ...old];
                });
                
                // Invalidate to ensure consistency
                queryClient.invalidateQueries({ queryKey: ['/api/notifications'] });
              }
              break;
              
            case 'keepalive':
              // Connection is alive
              break;
              
            default:
              console.log('Unknown SSE message type:', message);
          }
        } catch (error) {
          console.error('Error parsing SSE message:', error, event.data);
        }
      };

      eventSource.onerror = (error) => {
        console.error('SSE connection error:', error);
        setIsConnected(false);
        setConnectionError('Connection lost');
        
        // Close current connection
        if (eventSourceRef.current) {
          eventSourceRef.current.close();
          eventSourceRef.current = null;
        }
        
        // Attempt to reconnect with exponential backoff
        if (reconnectAttemptsRef.current < maxReconnectAttempts) {
          const delay = baseReconnectDelay * Math.pow(2, reconnectAttemptsRef.current);
          reconnectAttemptsRef.current++;
          
          console.log(`Attempting to reconnect in ${delay}ms (attempt ${reconnectAttemptsRef.current})`);
          
          reconnectTimeoutRef.current = setTimeout(() => {
            establishSSEConnection();
          }, delay);
        } else {
          console.error('Max reconnection attempts reached');
          setConnectionError('Failed to connect after multiple attempts');
        }
      };

    } catch (error) {
      console.error('Failed to establish SSE connection:', error);
      setConnectionError('Failed to establish connection');
      setIsConnected(false);
      setIsReconnecting(false);
    }
  }, [currentUser, queryClient, showNotificationToast]);

  const closeSSEConnection = useCallback(() => {
    if (reconnectTimeoutRef.current) {
      clearTimeout(reconnectTimeoutRef.current);
    }
    
    if (eventSourceRef.current) {
      eventSourceRef.current.close();
      eventSourceRef.current = null;
    }
    
    setIsConnected(false);
    setIsReconnecting(false);
    reconnectAttemptsRef.current = 0;
  }, []);

  // Establish connection when user is authenticated
  useEffect(() => {
    if (currentUser) {
      establishSSEConnection();
    } else {
      closeSSEConnection();
    }

    return () => {
      closeSSEConnection();
    };
  }, [currentUser, establishSSEConnection, closeSSEConnection]);

  // Cleanup on unmount
  useEffect(() => {
    return () => {
      closeSSEConnection();
    };
  }, [closeSSEConnection]);

  const contextValue: NotificationContextType = {
    notifications,
    unreadCount,
    isConnected,
    isReconnecting,
    connectionError,
  };

  return (
    <NotificationContext.Provider value={contextValue}>
      {children}
    </NotificationContext.Provider>
  );
}