import React from 'react';
import { useMutation, useQueryClient } from '@tanstack/react-query';
import { Bell, Check, X, Wifi, WifiOff } from 'lucide-react';
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuTrigger,
} from '@/components/ui/dropdown-menu';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { useAuth } from '@/contexts/AuthContext';
import { useNotifications } from '@/components/NotificationsProvider';
import { apiRequest } from '@/lib/queryClient';
import { formatDistanceToNow } from 'date-fns';

interface Notification {
  id: string;
  userId: string;
  message: string;
  type: 'info' | 'success' | 'warning' | 'error' | 'pro_welcome' | 'system';
  isRead: boolean;
  createdAt: string;
}

export function NotificationsBell() {
  const { currentUser } = useAuth();
  const queryClient = useQueryClient();
  const { notifications, unreadCount, isConnected, isReconnecting, connectionError } = useNotifications();

  // Mark notification as read mutation
  const markAsReadMutation = useMutation({
    mutationFn: async (notificationId: string) => {
      return apiRequest(`/api/notifications/mark-read/${notificationId}`, {
        method: 'POST',
      });
    },
    onSuccess: () => {
      // Invalidate notifications to trigger refetch
      queryClient.invalidateQueries({ queryKey: ['/api/notifications'] });
    },
    onError: (error) => {
      console.error('Failed to mark notification as read:', error);
    },
  });

  const handleMarkAsRead = (notificationId: string, event: React.MouseEvent) => {
    event.stopPropagation();
    markAsReadMutation.mutate(notificationId);
  };

  // Connection status indicator  
  const getConnectionIndicator = () => {
    if (isReconnecting) {
      return <WifiOff className="h-3 w-3 text-yellow-500 animate-pulse" />;
    }
    if (!isConnected) {
      return <WifiOff className="h-3 w-3 text-red-500" />;
    }
    if (isConnected) {
      return <Wifi className="h-3 w-3 text-green-500" />;
    }
    return null;
  };

  const getNotificationTypeColor = (type: string) => {
    switch (type) {
      case 'success':
        return 'bg-green-50 dark:bg-green-950 border-green-200 dark:border-green-800';
      case 'warning':
        return 'bg-yellow-50 dark:bg-yellow-950 border-yellow-200 dark:border-yellow-800';
      case 'error':
        return 'bg-red-50 dark:bg-red-950 border-red-200 dark:border-red-800';
      case 'pro_welcome':
        return 'bg-purple-50 dark:bg-purple-950 border-purple-200 dark:border-purple-800';
      default:
        return 'bg-blue-50 dark:bg-blue-950 border-blue-200 dark:border-blue-800';
    }
  };

  const getNotificationIcon = (type: string) => {
    switch (type) {
      case 'success':
        return <Check className="h-4 w-4 text-green-600 dark:text-green-400" />;
      case 'warning':
        return <Bell className="h-4 w-4 text-yellow-600 dark:text-yellow-400" />;
      case 'error':
        return <X className="h-4 w-4 text-red-600 dark:text-red-400" />;
      default:
        return <Bell className="h-4 w-4 text-blue-600 dark:text-blue-400" />;
    }
  };

  if (!currentUser) {
    return null;
  }

  return (
    <DropdownMenu>
      <DropdownMenuTrigger asChild>
        <Button
          variant="ghost"
          size="sm"
          className="relative p-2 text-muted-foreground hover:text-foreground rounded-lg hover:bg-muted transition-colors"
          data-testid="button-notifications-bell"
        >
          <div className="flex items-center gap-1">
            <Bell className="h-4 w-4" />
            {getConnectionIndicator()}
          </div>
          {unreadCount > 0 && (
            <Badge
              variant="destructive"
              className="absolute -top-1 -right-1 h-5 w-5 rounded-full p-0 flex items-center justify-center text-xs"
              data-testid="badge-notifications-count"
            >
              {unreadCount > 99 ? '99+' : unreadCount}
            </Badge>
          )}
        </Button>
      </DropdownMenuTrigger>
      
      <DropdownMenuContent 
        align="end" 
        sideOffset={5}
        className="w-80 max-h-96 overflow-y-auto"
        data-testid="dropdown-notifications"
      >
        <div className="px-3 py-2 border-b border-border">
          <div className="flex items-center justify-between">
            <h3 className="font-semibold text-sm text-foreground">
              Notifications
              {unreadCount > 0 && (
                <Badge variant="secondary" className="ml-2 text-xs">
                  {unreadCount} unread
                </Badge>
              )}
            </h3>
            <div className="flex items-center gap-1" data-testid="connection-status">
              {getConnectionIndicator()}
              <span className="text-xs text-muted-foreground">
                {isReconnecting ? 'Reconnecting...' : isConnected ? 'Live' : connectionError || 'Disconnected'}
              </span>
            </div>
          </div>
        </div>

        <div className="max-h-80 overflow-y-auto">
          {notifications.length === 0 ? (
            <div className="p-4 text-center text-muted-foreground text-sm">
              No notifications yet
            </div>
          ) : (
            <div className="py-1">
              {notifications.map((notification) => (
                <div
                  key={notification.id}
                  className={`px-3 py-3 border-b border-border last:border-b-0 cursor-default transition-colors ${
                    !notification.isRead 
                      ? `${getNotificationTypeColor(notification.type)} hover:opacity-80` 
                      : 'hover:bg-muted'
                  }`}
                  data-testid={`notification-item-${notification.id}`}
                >
                  <div className="flex items-start justify-between gap-2">
                    <div className="flex items-start gap-2 flex-1 min-w-0">
                      <div className="flex-shrink-0 mt-0.5">
                        {getNotificationIcon(notification.type)}
                      </div>
                      <div className="flex-1 min-w-0">
                        <p className="text-sm text-foreground leading-5 break-words">
                          {notification.message}
                        </p>
                        <p className="text-xs text-muted-foreground mt-1">
                          {formatDistanceToNow(new Date(notification.createdAt), { addSuffix: true })}
                        </p>
                      </div>
                    </div>
                    {!notification.isRead && (
                      <Button
                        variant="ghost"
                        size="sm"
                        className="p-1 h-6 w-6 text-muted-foreground hover:text-foreground"
                        onClick={(e) => handleMarkAsRead(notification.id, e)}
                        disabled={markAsReadMutation.isPending}
                        data-testid={`button-mark-read-${notification.id}`}
                      >
                        <Check className="h-3 w-3" />
                      </Button>
                    )}
                  </div>
                </div>
              ))}
            </div>
          )}
        </div>

        {notifications.length > 0 && (
          <div className="px-3 py-2 border-t border-border">
            <Button
              variant="ghost"
              size="sm"
              className="w-full text-xs text-muted-foreground hover:text-foreground"
              onClick={() => {
                // Future: Navigate to full notifications page
                console.log('Navigate to full notifications page');
              }}
              data-testid="button-view-all-notifications"
            >
              View all notifications
            </Button>
          </div>
        )}
      </DropdownMenuContent>
    </DropdownMenu>
  );
}