import { useState } from "react";
import { Link } from "wouter";
import { ChevronUp, Menu } from "lucide-react";
import { Button } from "@/components/ui/button";
import { useLogo } from "@/hooks/useLogo";

// Import page content components
import GlobalSlider from "@/components/GlobalSlider";
import SearchBar from "@/components/SearchBar";
import FeatureTiles from "@/components/FeatureTiles";
import Footer from "@/components/Footer";
import { homeSlides } from "@/data/homeSlides";

export default function MobileScrollLayout() {
  const [showNav, setShowNav] = useState(false);
  const logoSrc = useLogo();

  const scrollToSection = (sectionId: string) => {
    const element = document.getElementById(sectionId);
    if (element) {
      element.scrollIntoView({ behavior: 'smooth' });
      setShowNav(false);
    }
  };

  const scrollToTop = () => {
    window.scrollTo({ top: 0, behavior: 'smooth' });
  };

  return (
    <div className="min-h-screen bg-background">
      {/* Fixed Mobile Header */}
      <header className="fixed top-0 left-0 right-0 z-50 bg-white/95 dark:bg-gray-900/95 backdrop-blur-sm border-b shadow-sm">
        <div className="flex items-center justify-between p-4">
          <img 
            src={logoSrc} 
            alt="IBrandBiz Logo" 
            className="h-12 w-auto"
            onClick={scrollToTop}
          />
          <Button
            variant="ghost"
            size="sm"
            onClick={() => setShowNav(!showNav)}
            className="p-2"
          >
            <Menu className="h-5 w-5" />
          </Button>
        </div>
        
        {/* Mobile Navigation Menu */}
        {showNav && (
          <div className="absolute top-full left-0 right-0 bg-white dark:bg-gray-900 border-t shadow-lg">
            <nav className="p-4 space-y-3">
              <button 
                onClick={() => scrollToSection('home')}
                className="block w-full text-left py-2 px-3 text-gray-700 dark:text-gray-300 hover:bg-gray-50 dark:hover:bg-gray-800 rounded"
              >
                Home
              </button>
              <button 
                onClick={() => scrollToSection('services')}
                className="block w-full text-left py-2 px-3 text-gray-700 dark:text-gray-300 hover:bg-gray-50 dark:hover:bg-gray-800 rounded"
              >
                Services
              </button>
              <button 
                onClick={() => scrollToSection('products')}
                className="block w-full text-left py-2 px-3 text-gray-700 dark:text-gray-300 hover:bg-gray-50 dark:hover:bg-gray-800 rounded"
              >
                Products
              </button>
              <button 
                onClick={() => scrollToSection('pricing')}
                className="block w-full text-left py-2 px-3 text-gray-700 dark:text-gray-300 hover:bg-gray-50 dark:hover:bg-gray-800 rounded"
              >
                Pricing
              </button>
              <button 
                onClick={() => scrollToSection('resources')}
                className="block w-full text-left py-2 px-3 text-gray-700 dark:text-gray-300 hover:bg-gray-50 dark:hover:bg-gray-800 rounded"
              >
                Resources
              </button>
              <button 
                onClick={() => scrollToSection('faq')}
                className="block w-full text-left py-2 px-3 text-gray-700 dark:text-gray-300 hover:bg-gray-50 dark:hover:bg-gray-800 rounded"
              >
                FAQ
              </button>
              <hr className="my-2" />
              <Link href="/signup">
                <Button className="w-full bg-primary hover:bg-primary/90 text-white">
                  Get Started
                </Button>
              </Link>
            </nav>
          </div>
        )}
      </header>

      {/* Main Content - Long Scroll */}
      <main className="pt-20"> {/* Account for fixed header */}
        
        {/* Home Section */}
        <section id="home" className="min-h-screen">
          <div className="pt-4">
            <GlobalSlider 
              slides={homeSlides.filter(slide => slide.enabled !== false)} 
              autoPlay={true} 
              interval={10000} 
            />
          </div>
          
          <div className="bg-gray-50 dark:bg-gray-900">
            <SearchBar />
          </div>
          
          <div className="bg-white dark:bg-gray-800">
            <FeatureTiles />
          </div>
          
          {/* Value Proposition */}
          <div className="bg-gradient-to-br from-primary/5 to-accent/5 py-16">
            <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 text-center">
              <h2 className="text-3xl md:text-4xl font-bold text-gray-900 dark:text-white mb-6">
                Why Choose IBrandBiz?
              </h2>
              <div className="grid grid-cols-1 md:grid-cols-3 gap-8">
                <div className="flex flex-col items-center">
                  <div className="w-16 h-16 bg-white border-2 border-accent rounded-full flex items-center justify-center mb-4">
                    <span className="text-2xl">🤖</span>
                  </div>
                  <h3 className="text-xl font-semibold mb-2">AI-Powered</h3>
                  <p className="text-muted-foreground">
                    Advanced AI algorithms create unique, memorable names and designs tailored to your business.
                  </p>
                </div>
                <div className="flex flex-col items-center">
                  <div className="w-16 h-16 bg-white border-2 border-primary rounded-full flex items-center justify-center mb-4">
                    <span className="text-2xl">⚡</span>
                  </div>
                  <h3 className="text-xl font-semibold mb-2">Instant Results</h3>
                  <p className="text-muted-foreground">
                    Get professional-quality brand assets in minutes, not weeks. No waiting, no delays.
                  </p>
                </div>
                <div className="flex flex-col items-center">
                  <div className="w-16 h-16 bg-white border-2 border-secondary rounded-full flex items-center justify-center mb-4">
                    <span className="text-2xl">🎯</span>
                  </div>
                  <h3 className="text-xl font-semibold mb-2">All-in-One</h3>
                  <p className="text-muted-foreground">
                    From business names to domains to complete brand kits — everything you need in one place.
                  </p>
                </div>
              </div>
              
              {/* Stats */}
              <div className="grid grid-cols-2 md:grid-cols-4 gap-6 mt-12 pt-8 border-t border-gray-200 dark:border-gray-600">
                <div className="text-center">
                  <div className="text-3xl font-bold text-primary">10k+</div>
                  <div className="text-muted-foreground">Names Generated</div>
                </div>
                <div className="text-center">
                  <div className="text-3xl font-bold text-primary">5k+</div>
                  <div className="text-muted-foreground">Domains Registered</div>
                </div>
                <div className="text-center">
                  <div className="text-3xl font-bold text-primary">2k+</div>
                  <div className="text-muted-foreground">Brand Kits Created</div>
                </div>
                <div className="text-center">
                  <div className="text-3xl font-bold text-primary">3+</div>
                  <div className="text-muted-foreground">Years Experience</div>
                </div>
              </div>
            </div>
          </div>
        </section>

        {/* Services Section */}
        <section id="services" className="min-h-screen bg-white dark:bg-gray-800 py-16">
          <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
            <div className="text-center mb-12">
              <h2 className="text-3xl md:text-4xl font-bold text-gray-900 dark:text-white mb-4">
                Our Services
              </h2>
              <p className="text-lg text-muted-foreground max-w-3xl mx-auto">
                From naming to slogans and domains, we give you everything you need to launch and grow your brand with confidence.
              </p>
            </div>
            
            {/* Services content will be added here */}
            <div className="text-center">
              <p className="text-muted-foreground mb-8">
                Professional services to build your brand from the ground up.
              </p>
              <Link href="/services">
                <Button size="lg">
                  View All Services
                </Button>
              </Link>
            </div>
          </div>
        </section>

        {/* Products Section */}
        <section id="products" className="min-h-screen bg-gray-50 dark:bg-gray-900 py-16">
          <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
            <div className="text-center mb-12">
              <h2 className="text-3xl md:text-4xl font-bold text-gray-900 dark:text-white mb-4">
                Our Products
              </h2>
              <p className="text-lg text-muted-foreground max-w-3xl mx-auto">
                Everything you need to build your brand, from identity packages to business plans.
              </p>
            </div>
            
            <div className="text-center">
              <p className="text-muted-foreground mb-8">
                Complete digital tools and templates for your business success.
              </p>
              <Link href="/products">
                <Button size="lg">
                  Explore Products
                </Button>
              </Link>
            </div>
          </div>
        </section>

        {/* Pricing Section */}
        <section id="pricing" className="min-h-screen bg-white dark:bg-gray-800 py-16">
          <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
            <div className="text-center mb-12">
              <h2 className="text-3xl md:text-4xl font-bold text-gray-900 dark:text-white mb-4">
                Simple, Transparent Pricing
              </h2>
              <p className="text-lg text-muted-foreground max-w-3xl mx-auto">
                Choose the perfect plan for your business needs.
              </p>
            </div>
            
            <div className="text-center">
              <p className="text-muted-foreground mb-8">
                Start free, upgrade when you're ready to grow.
              </p>
              <Link href="/pricing/business-development">
                <Button size="lg">
                  View Pricing Plans
                </Button>
              </Link>
            </div>
          </div>
        </section>

        {/* Resources Section */}
        <section id="resources" className="min-h-screen bg-gray-50 dark:bg-gray-900 py-16">
          <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
            <div className="text-center mb-12">
              <h2 className="text-3xl md:text-4xl font-bold text-gray-900 dark:text-white mb-4">
                Resources & Learning
              </h2>
              <p className="text-lg text-muted-foreground max-w-3xl mx-auto">
                Guides, tips, and insights to help you build a successful brand.
              </p>
            </div>
            
            <div className="text-center">
              <p className="text-muted-foreground mb-8">
                Learn from experts and grow your business knowledge.
              </p>
              <Link href="/resources">
                <Button size="lg">
                  Browse Resources
                </Button>
              </Link>
            </div>
          </div>
        </section>

        {/* FAQ Section */}
        <section id="faq" className="min-h-screen bg-white dark:bg-gray-800 py-16">
          <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
            <div className="text-center mb-12">
              <h2 className="text-3xl md:text-4xl font-bold text-gray-900 dark:text-white mb-4">
                Frequently Asked Questions
              </h2>
              <p className="text-lg text-muted-foreground max-w-3xl mx-auto">
                Get answers to common questions about our platform and services.
              </p>
            </div>
            
            <div className="text-center">
              <p className="text-muted-foreground mb-8">
                Find quick answers to help you get started faster.
              </p>
              <Link href="/faq">
                <Button size="lg">
                  View All FAQs
                </Button>
              </Link>
            </div>
          </div>
        </section>

      </main>

      {/* Footer */}
      <Footer />

      {/* Floating Back to Top Button */}
      <Button
        onClick={scrollToTop}
        className="fixed bottom-6 right-6 z-40 w-12 h-12 rounded-full shadow-lg bg-primary hover:bg-primary/90 text-white"
        size="sm"
      >
        <ChevronUp className="h-5 w-5" />
      </Button>
    </div>
  );
}