import { Link } from "wouter";
import { Button } from "@/components/ui/button";
import { Menu, X } from "lucide-react";
import { useState } from "react";
import { useAuth } from "@/contexts/AuthContext";
import { AuthModal } from "@/components/AuthModal";
import { useLogo } from "@/hooks/useLogo";

export default function HeaderNav() {
  const [isMenuOpen, setIsMenuOpen] = useState(false);
  const [showAuthModal, setShowAuthModal] = useState(false);
  const { currentUser, logout } = useAuth();
  const logoSrc = useLogo();

  const navigationItems = [
    { label: "Home", href: "/" },
  ];

  const servicesItems = [
    { label: "Services Overview", href: "/services" },
  ];

  const productsItems = [
    { label: "Products Overview", href: "/products" },
    { label: "Website Templates", href: "/website-templates" },
    { label: "Social Media Kits", href: "/social-kits" },
  ];

  return (
    <header className="bg-white dark:bg-gray-900 mt-8">
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        <div className="flex justify-between items-center h-12">
          {/* Logo */}
          <Link href="/">
            <div className="flex items-center space-x-2 cursor-pointer" data-testid="link-logo">
              <img 
                src={logoSrc} 
                alt="IBrandBiz Logo" 
                className="h-[81px] w-auto"
              />
            </div>
          </Link>

          {/* Desktop Navigation */}
          <nav className="hidden md:flex items-center space-x-6">
            {navigationItems.map((item) => (
              <Link key={item.href} href={item.href}>
                <div 
                  className="flex items-center h-6 text-sm text-gray-600 dark:text-gray-300 hover:text-gray-900 dark:hover:text-white transition-colors cursor-pointer"
                  data-testid={`link-nav-${item.label.toLowerCase().replace(/\s+/g, '-')}`}
                >
                  {item.label}
                </div>
              </Link>
            ))}
            
            {/* Services Link */}
            <Link href="/services">
              <div 
                className="flex items-center h-6 text-sm text-gray-600 dark:text-gray-300 hover:text-gray-900 dark:hover:text-white transition-colors cursor-pointer"
                data-testid="link-nav-services"
              >
                Services
              </div>
            </Link>

            {/* Products Link */}
            <Link href="/products">
              <div 
                className="flex items-center h-6 text-sm text-gray-600 dark:text-gray-300 hover:text-gray-900 dark:hover:text-white transition-colors cursor-pointer"
                data-testid="link-nav-products"
              >
                Products
              </div>
            </Link>

            {/* Pricing Dropdown */}
            <div className="relative group">
              <div 
                className="flex items-center h-6 text-sm text-gray-600 dark:text-gray-300 hover:text-gray-900 dark:hover:text-white transition-colors cursor-pointer"
                data-testid="dropdown-pricing"
              >
                Pricing
                <svg className="w-4 h-4 ml-1" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 9l-7 7-7-7" />
                </svg>
              </div>
              
              {/* Dropdown Menu */}
              <div className="absolute left-0 mt-2 w-56 bg-white dark:bg-gray-800 rounded-md shadow-lg border border-gray-200 dark:border-gray-600 opacity-0 invisible group-hover:opacity-100 group-hover:visible transition-all duration-200 z-50">
                <div className="py-1">
                  <Link href="/pricing/business-development">
                    <div className="block px-4 py-2 text-sm text-gray-700 dark:text-gray-300 hover:bg-gray-100 dark:hover:bg-gray-700 cursor-pointer" data-testid="link-pricing-business-development">
                      Business Development
                    </div>
                  </Link>
                  <Link href="/pricing/web-services">
                    <div className="block px-4 py-2 text-sm text-gray-700 dark:text-gray-300 hover:bg-gray-100 dark:hover:bg-gray-700 cursor-pointer" data-testid="link-pricing-web-services">
                      Web Services
                    </div>
                  </Link>
                  <Link href="/pricing/addons">
                    <div className="block px-4 py-2 text-sm text-gray-700 dark:text-gray-300 hover:bg-gray-100 dark:hover:bg-gray-700 cursor-pointer" data-testid="link-pricing-addons">
                      Web Add-Ons
                    </div>
                  </Link>
                </div>
              </div>
            </div>

            {/* Resources Link */}
            <Link href="/resources">
              <div 
                className="flex items-center h-6 text-sm text-gray-600 dark:text-gray-300 hover:text-gray-900 dark:hover:text-white transition-colors cursor-pointer"
                data-testid="link-nav-resources"
              >
                Resources
              </div>
            </Link>

            {/* FAQ Link */}
            <Link href="/faq">
              <div 
                className="flex items-center h-6 text-sm text-gray-600 dark:text-gray-300 hover:text-gray-900 dark:hover:text-white transition-colors cursor-pointer"
                data-testid="link-nav-faq"
              >
                FAQ
              </div>
            </Link>
            
            {/* Dashboard Link - Always visible */}
            <Link href="/dashboard">
              <div 
                className="flex items-center h-6 text-sm text-gray-600 dark:text-gray-300 hover:text-gray-900 dark:hover:text-white transition-colors cursor-pointer"
                data-testid="link-nav-dashboard"
              >
                Dashboard
              </div>
            </Link>
            
            {currentUser ? (
              <Button 
                className="text-white font-bold px-3" 
                style={{backgroundColor: '#ff8800'}} 
                onMouseEnter={(e) => ((e.target as HTMLButtonElement).style.backgroundColor = '#e67700')} 
                onMouseLeave={(e) => ((e.target as HTMLButtonElement).style.backgroundColor = '#ff8800')}
                onClick={() => logout()}
                data-testid="button-logout"
              >
                Logout
              </Button>
            ) : (
              <Button 
                className="text-white font-bold px-3" 
                style={{backgroundColor: '#00C851'}} 
                onMouseEnter={(e) => ((e.target as HTMLButtonElement).style.backgroundColor = '#00B347')} 
                onMouseLeave={(e) => ((e.target as HTMLButtonElement).style.backgroundColor = '#00C851')}
                onClick={() => setShowAuthModal(true)}
                data-testid="button-login"
              >
                Login
              </Button>
            )}
          </nav>

          {/* Mobile menu button */}
          <div className="md:hidden">
            <Button
              variant="ghost"
              size="sm"
              onClick={() => setIsMenuOpen(!isMenuOpen)}
              data-testid="button-mobile-menu"
            >
              {isMenuOpen ? <X className="h-6 w-6" /> : <Menu className="h-6 w-6" />}
            </Button>
          </div>
        </div>

        {/* Mobile Navigation */}
        {isMenuOpen && (
          <div className="md:hidden bg-white dark:bg-gray-900 border-t">
            <nav className="py-3 space-y-1">
              {navigationItems.map((item) => (
                <Link key={item.href} href={item.href}>
                  <div 
                    className="block px-3 py-1 text-sm text-gray-600 dark:text-gray-300 hover:text-gray-900 dark:hover:text-white hover:bg-gray-50 dark:hover:bg-gray-800 transition-colors cursor-pointer"
                    data-testid={`link-mobile-${item.label.toLowerCase().replace(/\s+/g, '-')}`}
                    onClick={() => setIsMenuOpen(false)}
                  >
                    {item.label}
                  </div>
                </Link>
              ))}
              
              {/* Mobile Services */}
              <Link href="/services">
                <div 
                  className="block px-3 py-1 text-sm text-gray-600 dark:text-gray-300 hover:text-gray-900 dark:hover:text-white hover:bg-gray-50 dark:hover:bg-gray-800 transition-colors cursor-pointer"
                  data-testid="link-mobile-services"
                  onClick={() => setIsMenuOpen(false)}
                >
                  Services
                </div>
              </Link>
              
              {/* Mobile Products */}
              <Link href="/products">
                <div 
                  className="block px-3 py-1 text-sm text-gray-600 dark:text-gray-300 hover:text-gray-900 dark:hover:text-white hover:bg-gray-50 dark:hover:bg-gray-800 transition-colors cursor-pointer"
                  data-testid="link-mobile-products"
                  onClick={() => setIsMenuOpen(false)}
                >
                  Products
                </div>
              </Link>
              
              {/* Mobile Pricing */}
              <Link href="/pricing/business-development">
                <div 
                  className="block px-3 py-1 text-sm text-gray-600 dark:text-gray-300 hover:text-gray-900 dark:hover:text-white hover:bg-gray-50 dark:hover:bg-gray-800 transition-colors cursor-pointer"
                  data-testid="link-mobile-pricing"
                  onClick={() => setIsMenuOpen(false)}
                >
                  Pricing
                </div>
              </Link>
              
              {/* Mobile Resources */}
              <Link href="/resources">
                <div 
                  className="block px-3 py-1 text-sm text-gray-600 dark:text-gray-300 hover:text-gray-900 dark:hover:text-white hover:bg-gray-50 dark:hover:bg-gray-800 transition-colors cursor-pointer"
                  data-testid="link-mobile-resources"
                  onClick={() => setIsMenuOpen(false)}
                >
                  Resources
                </div>
              </Link>
              
              {/* Mobile FAQ */}
              <Link href="/faq">
                <div 
                  className="block px-3 py-1 text-sm text-gray-600 dark:text-gray-300 hover:text-gray-900 dark:hover:text-white hover:bg-gray-50 dark:hover:bg-gray-800 transition-colors cursor-pointer"
                  data-testid="link-mobile-faq"
                  onClick={() => setIsMenuOpen(false)}
                >
                  FAQ
                </div>
              </Link>
              
              {/* Mobile Dashboard - Always visible */}
              <Link href="/dashboard">
                <div 
                  className="block px-3 py-1 text-sm text-gray-600 dark:text-gray-300 hover:text-gray-900 dark:hover:text-white hover:bg-gray-50 dark:hover:bg-gray-800 transition-colors cursor-pointer"
                  data-testid="link-mobile-dashboard"
                  onClick={() => setIsMenuOpen(false)}
                >
                  Dashboard
                </div>
              </Link>
              
              <div className="px-3 pt-1">
                {currentUser ? (
                  <Button 
                    className="w-full text-white font-bold" 
                    style={{backgroundColor: '#ff8800'}} 
                    onMouseEnter={(e) => ((e.target as HTMLButtonElement).style.backgroundColor = '#e67700')} 
                    onMouseLeave={(e) => ((e.target as HTMLButtonElement).style.backgroundColor = '#ff8800')}
                    onClick={() => {
                      logout();
                      setIsMenuOpen(false);
                    }}
                    data-testid="button-mobile-logout"
                  >
                    Logout
                  </Button>
                ) : (
                  <Button 
                    className="w-full text-white font-bold" 
                    style={{backgroundColor: '#00C851'}} 
                    onMouseEnter={(e) => ((e.target as HTMLButtonElement).style.backgroundColor = '#00B347')} 
                    onMouseLeave={(e) => ((e.target as HTMLButtonElement).style.backgroundColor = '#00C851')}
                    onClick={() => {
                      setShowAuthModal(true);
                      setIsMenuOpen(false);
                    }}
                    data-testid="button-mobile-login"
                  >
                    Login
                  </Button>
                )}
              </div>
            </nav>
          </div>
        )}
      </div>
      
      {/* Auth Modal */}
      <AuthModal 
        isOpen={showAuthModal} 
        onClose={() => setShowAuthModal(false)} 
      />
    </header>
  );
}