import { useState, useEffect } from "react";
import { Button } from "@/components/ui/button";
import { ChevronLeft, ChevronRight } from "lucide-react";
import { HomeSlide } from "@/data/homeSlides";
import { Link } from "wouter";

interface GlobalSliderProps {
  slides: HomeSlide[];
  autoPlay?: boolean;
  interval?: number;
}

export default function GlobalSlider({ slides, autoPlay = true, interval = 6000 }: GlobalSliderProps) {
  const [currentIndex, setCurrentIndex] = useState(0);

  // Auto-play functionality
  useEffect(() => {
    if (!autoPlay || slides.length <= 1) return;

    const timer = setInterval(() => {
      setCurrentIndex((prevIndex) => (prevIndex + 1) % slides.length);
    }, interval);

    return () => clearInterval(timer);
  }, [autoPlay, interval, slides.length]);

  const goToPrevious = () => {
    setCurrentIndex((prevIndex) => (prevIndex - 1 + slides.length) % slides.length);
  };

  const goToNext = () => {
    setCurrentIndex((prevIndex) => (prevIndex + 1) % slides.length);
  };

  const goToSlide = (index: number) => {
    setCurrentIndex(index);
  };

  if (!slides.length) return null;

  const currentSlide = slides[currentIndex];

  return (
    <div className="hero-slider relative overflow-hidden mt-6 mx-auto" style={{ width: '100%', maxWidth: '2048px', aspectRatio: '2048 / 632' }}>
      {/* Background Image */}
      <div className="absolute inset-0">
        <div
          className="bg-gray-300"
          style={{
            width: '100%',
            height: '100%',
            backgroundImage: `url(${currentSlide.imageUrl})`,
            backgroundSize: 'cover',
            backgroundPosition: 'center',
            objectFit: 'cover'
          }}
          data-testid={`slide-background-${currentSlide.id}`}
        />
      </div>

      {/* Left-weighted Gradient Overlay */}
      <div className="absolute inset-0 bg-gradient-to-r from-black/60 via-black/30 to-transparent pointer-events-none" />

      {/* Content Overlay */}
      <div className="absolute inset-0 flex items-center justify-start z-30">
        <div className="text-left max-w-xl pl-6 md:pl-16 lg:pl-20 pr-4 md:pr-8">
          <h1 
            className="text-xl md:text-3xl lg:text-4xl font-bold text-white mb-2 drop-shadow-lg whitespace-pre-line"
            data-testid={`slide-title-${currentSlide.id}`}
          >
            {currentSlide.title}
          </h1>
          {currentSlide.subtitle.includes('<div') ? (
            <div 
              className="text-xs md:text-sm lg:text-base text-white mb-4 drop-shadow-lg max-w-lg"
              data-testid={`slide-subtitle-${currentSlide.id}`}
              dangerouslySetInnerHTML={{ __html: currentSlide.subtitle }}
            />
          ) : (
            <p 
              className="text-xs md:text-sm lg:text-base text-white mb-4 drop-shadow-lg max-w-lg whitespace-pre-line"
              data-testid={`slide-subtitle-${currentSlide.id}`}
            >
              {currentSlide.subtitle}
            </p>
          )}
          {currentSlide.cta && (
            <Link href={currentSlide.cta.href}>
              <Button 
                className="text-white font-bold px-6 py-2 text-xs md:text-sm shadow-lg hover:shadow-xl transition-all duration-300" style={{backgroundColor: '#00C851'}} onMouseEnter={(e) => ((e.target as HTMLButtonElement).style.backgroundColor = '#00B347')} onMouseLeave={(e) => ((e.target as HTMLButtonElement).style.backgroundColor = '#00C851')}
                data-testid={`slide-cta-${currentSlide.id}`}
              >
                {currentSlide.cta.label}
              </Button>
            </Link>
          )}
        </div>
      </div>

      {/* Navigation Arrows */}
      {slides.length > 1 && (
        <>
          <Button
            variant="ghost"
            size="icon"
            className="absolute left-3 top-1/2 transform -translate-y-1/2 bg-white/20 hover:bg-white/30 text-white border-white/30 z-40"
            onClick={goToPrevious}
            data-testid="button-slider-prev"
            aria-label="Previous slide"
          >
            <ChevronLeft className="h-6 w-6" />
          </Button>
          <Button
            variant="ghost"
            size="icon"
            className="absolute right-3 top-1/2 transform -translate-y-1/2 bg-white/20 hover:bg-white/30 text-white border-white/30 z-40"
            onClick={goToNext}
            data-testid="button-slider-next"
            aria-label="Next slide"
          >
            <ChevronRight className="h-6 w-6" />
          </Button>
        </>
      )}

      {/* Pagination Dots */}
      {slides.length > 1 && (
        <div className="absolute bottom-4 left-1/2 transform -translate-x-1/2 flex space-x-2 z-40">
          {slides.map((_, index) => (
            <button
              key={index}
              className={`w-3 h-3 rounded-full transition-all duration-300 ${
                index === currentIndex 
                  ? "bg-white scale-110" 
                  : "bg-white/50 hover:bg-white/70"
              }`}
              onClick={() => goToSlide(index)}
              data-testid={`button-dot-${index}`}
            />
          ))}
        </div>
      )}
    </div>
  );
}