import React, { useMemo, useState } from "react";
import { Button } from "@/components/ui/button";
import { Loader2, Wand2, RotateCcw } from "lucide-react";
import { getGeneratedFontOptions } from "@/services/fonts/generator";

export type FontOption = {
  label: string;          // human name shown on the tile
  family: string;         // CSS font-family to apply (must match your loaded Google Font)
  group?: "Sans" | "Serif" | "Slab" | "Display" | "Script" | "Mono";
};

type Props = {
  value: string;
  onChange: (family: string) => void;
  sampleText: string;     // usually the brand name
  fonts?: FontOption[];   // optional override list
  height?: number;        // px of the grid area
};

const DEFAULT_FONTS: FontOption[] = [
  // --- Sans Serif (20) ---
  { label: "Inter", family: '"Inter", sans-serif', group: "Sans" },
  { label: "Manrope", family: '"Manrope", sans-serif', group: "Sans" },
  { label: "Montserrat", family: '"Montserrat", sans-serif', group: "Sans" },
  { label: "Poppins", family: '"Poppins", sans-serif', group: "Sans" },
  { label: "Roboto", family: '"Roboto", sans-serif', group: "Sans" },
  { label: "Open Sans", family: '"Open Sans", sans-serif', group: "Sans" },
  { label: "Lato", family: '"Lato", sans-serif', group: "Sans" },
  { label: "Work Sans", family: '"Work Sans", sans-serif', group: "Sans" },
  { label: "Nunito", family: '"Nunito", sans-serif', group: "Sans" },
  { label: "Raleway", family: '"Raleway", sans-serif', group: "Sans" },
  { label: "Rubik", family: '"Rubik", sans-serif', group: "Sans" },
  { label: "Barlow", family: '"Barlow", sans-serif', group: "Sans" },
  { label: "Heebo", family: '"Heebo", sans-serif', group: "Sans" },
  { label: "Archivo", family: '"Archivo", sans-serif', group: "Sans" },
  { label: "Catamaran", family: '"Catamaran", sans-serif', group: "Sans" },
  { label: "Lexend", family: '"Lexend", sans-serif', group: "Sans" },
  { label: "Titillium Web", family: '"Titillium Web", sans-serif', group: "Sans" },
  { label: "Chivo", family: '"Chivo", sans-serif', group: "Sans" },
  { label: "Overpass", family: '"Overpass", sans-serif', group: "Sans" },
  { label: "Ubuntu", family: '"Ubuntu", sans-serif', group: "Sans" },

  // --- Serif (10) ---
  { label: "Playfair Display", family: '"Playfair Display", serif', group: "Serif" },
  { label: "Merriweather", family: '"Merriweather", serif', group: "Serif" },
  { label: "Cormorant Garamond", family: '"Cormorant Garamond", serif', group: "Serif" },
  { label: "PT Serif", family: '"PT Serif", serif', group: "Serif" },
  { label: "Domine", family: '"Domine", serif', group: "Serif" },
  { label: "Zilla Slab", family: '"Zilla Slab", serif', group: "Serif" },
  { label: "Crimson Text", family: '"Crimson Text", serif', group: "Serif" },
  { label: "Lora", family: '"Lora", serif', group: "Serif" },
  { label: "Spectral", family: '"Spectral", serif', group: "Serif" },
  { label: "Vollkorn", family: '"Vollkorn", serif', group: "Serif" },

  // --- Slab Serif (5) ---
  { label: "Arvo", family: '"Arvo", serif', group: "Slab" },
  { label: "Roboto Slab", family: '"Roboto Slab", serif', group: "Slab" },
  { label: "Noto Serif Display", family: '"Noto Serif Display", serif', group: "Slab" },
  { label: "Alegreya", family: '"Alegreya", serif', group: "Slab" },
  { label: "Tinos", family: '"Tinos", serif', group: "Slab" },

  // --- Display / Tech / Condensed (10) ---
  { label: "Bebas Neue", family: '"Bebas Neue", sans-serif', group: "Display" },
  { label: "Oswald", family: '"Oswald", sans-serif', group: "Display" },
  { label: "Orbitron", family: '"Orbitron", sans-serif', group: "Display" },
  { label: "Varela Round", family: '"Varela Round", sans-serif', group: "Display" },
  { label: "Play", family: '"Play", sans-serif', group: "Display" },
  { label: "Exo 2", family: '"Exo 2", sans-serif', group: "Display" },
  { label: "Righteous", family: '"Righteous", cursive', group: "Display" },
  { label: "Anton", family: '"Anton", sans-serif', group: "Display" },
  { label: "Fjalla One", family: '"Fjalla One", sans-serif', group: "Display" },
  { label: "Bungee", family: '"Bungee", cursive', group: "Display" },

  // --- Script / Handwriting (10) ---
  { label: "Pacifico", family: '"Pacifico", cursive', group: "Script" },
  { label: "Dancing Script", family: '"Dancing Script", cursive', group: "Script" },
  { label: "Great Vibes", family: '"Great Vibes", cursive', group: "Script" },
  { label: "Permanent Marker", family: '"Permanent Marker", cursive', group: "Script" },
  { label: "Amatic SC", family: '"Amatic SC", cursive', group: "Script" },
  { label: "Sacramento", family: '"Sacramento", cursive', group: "Script" },
  { label: "Indie Flower", family: '"Indie Flower", cursive', group: "Script" },
  { label: "Shadows Into Light", family: '"Shadows Into Light", cursive', group: "Script" },
  { label: "Courgette", family: '"Courgette", cursive', group: "Script" },
  { label: "Kaushan Script", family: '"Kaushan Script", cursive', group: "Script" },
];

export function FontPicker({
  value,
  onChange,
  sampleText,
  fonts = DEFAULT_FONTS,
  height = 320,
}: Props) {
  const [query, setQuery] = useState("");
  const [group, setGroup] = useState<FontOption["group"] | "All">("All");
  const [generatedFonts, setGeneratedFonts] = useState<FontOption[]>([]);
  const [isGenerating, setIsGenerating] = useState(false);
  const [generationError, setGenerationError] = useState<string | null>(null);
  const [showGenerated, setShowGenerated] = useState(false);

  // Generate fonts using AI
  const generateFonts = async () => {
    if (isGenerating) return;
    
    setIsGenerating(true);
    setGenerationError(null);
    
    try {
      const generatedFontOptions = await getGeneratedFontOptions();
      setGeneratedFonts(generatedFontOptions);
      setShowGenerated(true);
      console.log(`Generated ${generatedFontOptions.length} font options`);
    } catch (error: any) {
      console.error('Font generation failed:', error);
      setGenerationError(error.message || 'Failed to generate fonts');
    } finally {
      setIsGenerating(false);
    }
  };

  // Use generated fonts if available and enabled, otherwise use default fonts
  const activeFonts = showGenerated && generatedFonts.length > 0 ? generatedFonts : fonts;

  const filtered = useMemo(() => {
    const q = query.trim().toLowerCase();
    return activeFonts.filter((f) => {
      const inGroup = group === "All" || f.group === group;
      const match = !q || f.label.toLowerCase().includes(q);
      return inGroup && match;
    });
  }, [activeFonts, query, group]);

  return (
    <div className="space-y-2">
      <div className="flex items-center gap-2">
        <input
          className="px-3 py-2 rounded-md border flex-1"
          placeholder="Search fonts…"
          value={query}
          onChange={(e) => setQuery(e.target.value)}
          data-testid="input-font-search"
        />
        <select
          className="px-3 py-2 rounded-md border"
          value={group}
          onChange={(e) => setGroup(e.target.value as any)}
          data-testid="select-font-group"
        >
          <option value="All">All</option>
          <option value="Sans">Sans</option>
          <option value="Serif">Serif</option>
          <option value="Slab">Slab</option>
          <option value="Display">Display</option>
          <option value="Script">Script</option>
          <option value="Mono">Mono</option>
        </select>
      </div>

      {/* Font Generation Controls */}
      <div className="flex items-center gap-2 p-3 bg-gray-50 dark:bg-gray-800 rounded-lg">
        <div className="flex-1">
          <div className="flex items-center gap-2">
            <Wand2 className="w-4 h-4 text-blue-600" />
            <span className="text-sm font-medium">
              AI-Generated Professional Fonts 
            </span>
            {showGenerated && generatedFonts.length > 0 && (
              <span className="text-xs text-green-600 bg-green-100 px-2 py-1 rounded">
                {generatedFonts.length} fonts loaded
              </span>
            )}
          </div>
          <p className="text-xs text-gray-600 dark:text-gray-400 mt-1">
            Generate 120 professional font options using AI
          </p>
        </div>
        
        <div className="flex items-center gap-2">
          {showGenerated && generatedFonts.length > 0 && (
            <Button
              variant="outline"
              size="sm"
              onClick={() => setShowGenerated(false)}
              disabled={isGenerating}
              data-testid="button-use-default-fonts"
            >
              <RotateCcw className="w-3 h-3 mr-1" />
              Use Default
            </Button>
          )}
          
          <Button
            size="sm"
            onClick={generateFonts}
            disabled={isGenerating}
            className="min-w-[100px]"
            data-testid="button-generate-fonts"
          >
            {isGenerating ? (
              <>
                <Loader2 className="w-3 h-3 mr-1 animate-spin" />
                Generating...
              </>
            ) : (
              <>
                <Wand2 className="w-3 h-3 mr-1" />
                {generatedFonts.length > 0 ? 'Regenerate' : 'Generate'}
              </>
            )}
          </Button>
        </div>
      </div>

      {/* Error Message */}
      {generationError && (
        <div className="p-3 bg-red-50 dark:bg-red-900/20 border border-red-200 dark:border-red-800 rounded-lg">
          <p className="text-sm text-red-600 dark:text-red-400">
            {generationError}
          </p>
        </div>
      )}

      {/* Active Font Set Indicator */}
      <div className="flex items-center justify-between text-xs text-gray-500">
        <span>
          Showing {filtered.length} of {activeFonts.length} fonts
          {showGenerated && generatedFonts.length > 0 ? ' (AI Generated)' : ' (Default Collection)'}
        </span>
        {showGenerated && generatedFonts.length > 0 && (
          <span className="text-blue-600 font-medium">✨ AI Enhanced</span>
        )}
      </div>

      <div
        className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-2 overflow-auto pr-1"
        style={{ maxHeight: height }}
        role="listbox"
        aria-label="Choose a font"
      >
        {filtered.map((f) => {
          const active = f.family === value;
          return (
            <button
              key={f.label}
              onClick={() => onChange(f.family)}
              className={`text-left rounded-lg border p-3 hover:bg-neutral-50 focus:outline-none focus:ring-2 focus:ring-neutral-300 ${
                active ? "border-neutral-900" : "border-neutral-200"
              }`}
              role="option"
              aria-selected={active}
              title={f.label}
            >
              <div className="text-[11px] text-neutral-500 mb-1">{f.label}</div>
              <div
                className="leading-snug"
                style={{ fontFamily: f.family, fontWeight: 700, fontSize: 28 }}
              >
                {sampleText || f.label}
              </div>
            </button>
          );
        })}
      </div>

      <div className="text-xs text-neutral-500">
        Tip: This list uses Google Fonts. Make sure the families are included in your
        page's link tags so they render correctly.
      </div>
    </div>
  );
}

export default FontPicker;