import { useState, useRef } from 'react';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Card, CardContent } from '@/components/ui/card';
import { Progress } from '@/components/ui/progress';
import { useToast } from '@/hooks/use-toast';
import { Upload, X, CheckCircle } from 'lucide-react';

interface FileUploaderProps {
  onUploadComplete?: (url: string) => void;
  accept?: string;
  maxSizeBytes?: number;
  className?: string;
  children?: React.ReactNode;
}

export function FileUploader({
  onUploadComplete,
  accept = "image/*",
  maxSizeBytes = 5 * 1024 * 1024, // 5MB
  className,
  children
}: FileUploaderProps) {
  const [uploading, setUploading] = useState(false);
  const [progress, setProgress] = useState(0);
  const [uploadedFile, setUploadedFile] = useState<{ name: string; url: string } | null>(null);
  const fileInputRef = useRef<HTMLInputElement>(null);
  const { toast } = useToast();

  const handleFileSelect = async (event: React.ChangeEvent<HTMLInputElement>) => {
    const file = event.target.files?.[0];
    if (!file) return;

    // Validate file size
    if (file.size > maxSizeBytes) {
      toast({
        title: "File too large",
        description: `File size must be less than ${Math.round(maxSizeBytes / (1024 * 1024))}MB`,
        variant: "destructive"
      });
      return;
    }

    await uploadFile(file);
  };

  const uploadFile = async (file: File) => {
    try {
      setUploading(true);
      setProgress(10);

      // Get upload URL from backend
      const response = await fetch('/api/upload-url', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Authorization': `Bearer ${await getAuthToken()}`
        },
        body: JSON.stringify({ fileName: file.name })
      });

      if (!response.ok) {
        throw new Error('Failed to get upload URL');
      }

      const { uploadURL } = await response.json();
      setProgress(30);

      // Upload file directly to storage
      const uploadResponse = await fetch(uploadURL, {
        method: 'PUT',
        body: file,
        headers: {
          'Content-Type': file.type
        }
      });

      if (!uploadResponse.ok) {
        throw new Error('Failed to upload file');
      }

      setProgress(80);

      // Get the public URL
      const publicURL = `/public-objects/${file.name}`;
      setProgress(100);

      setUploadedFile({ name: file.name, url: publicURL });
      onUploadComplete?.(publicURL);

      toast({
        title: "Upload successful",
        description: `${file.name} has been uploaded successfully`,
      });

    } catch (error) {
      console.error('Upload failed:', error);
      toast({
        title: "Upload failed",
        description: "There was an error uploading your file. Please try again.",
        variant: "destructive"
      });
    } finally {
      setUploading(false);
      setProgress(0);
    }
  };

  const getAuthToken = async () => {
    // Get Firebase ID token from the current user
    const auth = await import('firebase/auth');
    const { auth: firebaseAuth } = await import('@/lib/firebase');
    const user = auth.getAuth().currentUser;
    if (user) {
      return await user.getIdToken();
    }
    return '';
  };

  const handleRemove = () => {
    setUploadedFile(null);
    if (fileInputRef.current) {
      fileInputRef.current.value = '';
    }
  };

  return (
    <div className={className}>
      {!uploadedFile ? (
        <Card>
          <CardContent className="p-6">
            <div className="text-center">
              {children || (
                <>
                  <Upload className="mx-auto h-12 w-12 text-muted-foreground mb-4" />
                  <h3 className="text-lg font-semibold mb-2">Upload File</h3>
                  <p className="text-muted-foreground mb-4">
                    Drag and drop your file here, or click to browse
                  </p>
                </>
              )}
              
              <Input
                ref={fileInputRef}
                type="file"
                accept={accept}
                onChange={handleFileSelect}
                disabled={uploading}
                className="hidden"
                data-testid="file-input"
              />
              
              <Button
                onClick={() => fileInputRef.current?.click()}
                disabled={uploading}
                className="w-full"
                data-testid="upload-button"
              >
                {uploading ? 'Uploading...' : 'Choose File'}
              </Button>
              
              {uploading && (
                <div className="mt-4">
                  <Progress value={progress} className="mb-2" />
                  <p className="text-sm text-muted-foreground">{progress}% uploaded</p>
                </div>
              )}
            </div>
          </CardContent>
        </Card>
      ) : (
        <Card>
          <CardContent className="p-4">
            <div className="flex items-center justify-between">
              <div className="flex items-center space-x-3">
                <CheckCircle className="h-5 w-5 text-green-500" />
                <div>
                  <p className="font-medium">{uploadedFile.name}</p>
                  <p className="text-sm text-muted-foreground">Upload complete</p>
                </div>
              </div>
              <Button
                variant="ghost"
                size="sm"
                onClick={handleRemove}
                data-testid="remove-file"
              >
                <X className="h-4 w-4" />
              </Button>
            </div>
          </CardContent>
        </Card>
      )}
    </div>
  );
}