import { Link } from "wouter";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Wand2, Palette, Globe, Share2, ArrowRight, Server, Image, Briefcase, MessageSquare, FileText, Presentation, Boxes, Images, PieChart } from "lucide-react";
import { bulletCubeWhite } from '@/assets';

interface FeatureTile {
  id: string;
  title: string;
  description: string;
  icon: JSX.Element;
  href: string;
  gradient: string;
  stats?: string;
  cta?: { label: string; href: string };
  enabled?: boolean;
}

export default function FeatureTiles() {
  const mainSections: FeatureTile[] = [
    {
      id: "business-development",
      title: "Business Development",
      description: "Complete toolkit for building your business foundation from the ground up.",
      icon: <Wand2 className="h-8 w-8" />,
      href: "/business-development/brand-name-wizard",
      gradient: "from-blue-500 to-cyan-500",
      stats: "5 Essential Tools",
      cta: { label: "Start Building", href: "/business-development/brand-name-wizard" }
    },
    {
      id: "business-assets",
      title: "Business Assets",
      description: "Professional templates and design resources to elevate your brand.",
      icon: <Briefcase className="h-8 w-8" />,
      href: "/business-assets/templates/business-plan",
      gradient: "from-purple-500 to-pink-500",
      stats: "Premium Templates",
      cta: { label: "Explore Assets", href: "/business-assets/templates/business-plan" }
    },
    {
      id: "web-services",
      title: "Web Services",
      description: "Essential web solutions for your online presence, from domains to hosting.",
      icon: <Globe className="h-8 w-8" />,
      href: "/web-services/domains",
      gradient: "from-orange-500 to-red-500",
      stats: "Online Solutions",
      cta: { label: "Get Online", href: "/web-services/domains" }
    }
  ];

  // Section details for each main category
  const sectionDetails = {
    "business-development": {
      items: [
        { name: "Brand Name Wizard (Free)", desc: "AI-powered name generation for your brand", href: "/business-development/brand-name-wizard" },
        { name: "Slogan Generator (Free)", desc: "Create memorable taglines and slogans", href: "/business-development/slogans" },
        { name: "AI Logo Creator (Membership)", desc: "AI-powered logo generation with customization", href: "/business-development/ai-logo-creator" },
        { name: "Brand Kit (Membership)", desc: "Complete brand identity packages", href: "/business-development/brand-kit" },
        { name: "AI Business Plan Developer (Membership)", desc: "Professional business plan templates", href: "/business-development/business-plan-builder" }
      ]
    },
    "business-assets": {
      items: [
        { name: "Business Plan Templates (Free)", desc: "Professional business plan templates and formats", href: "/business-assets/templates/business-plan" },
        { name: "Logo Templates (Free)", desc: "Browse and customize professional logo templates", href: "/business-assets/logo-templates" }
      ]
    },
    "web-services": {
      items: [
        { name: "Domain Search (Paid Service)", desc: "Find and register the perfect domain", href: "/web-services/domains" },
        { name: "Web Hosting (Paid Service)", desc: "Reliable hosting solutions for your website", href: "/web-services/web-hosting" }
      ]
    }
  };

  return (
    <section className="w-full max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-20">
      <div className="text-center mb-16">
        <h2 className="text-3xl md:text-4xl font-bold text-gray-900 dark:text-white mb-4">
          Everything You Need to Build Your Brand
        </h2>
        <p className="text-base text-muted-foreground max-w-2xl mx-auto">
          From naming your business to creating stunning logos, we have all the tools to bring your brand to life.
        </p>
      </div>

      {/* Main Sections Grid */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-12 justify-items-center max-w-6xl mx-auto">
        {mainSections.map((section) => (
          <Card 
            key={section.id}
            className="group cursor-pointer border-0 shadow-lg hover:shadow-2xl transition-all duration-300 hover:-translate-y-2 bg-white dark:bg-gray-800 overflow-hidden w-full max-w-sm flex flex-col min-h-[600px]"
            data-testid={`card-section-${section.id}`}
          >
            {/* Header Section */}
            <Link href={section.href}>
              <CardHeader className="relative p-6">
                {/* Gradient background */}
                <div className={`absolute top-0 left-0 w-full h-20 bg-gradient-to-r ${section.gradient} opacity-10 group-hover:opacity-20 transition-opacity duration-300`} />
                
                {/* Icon */}
                <div className={`relative w-16 h-16 rounded-xl bg-gradient-to-r ${section.gradient} flex items-center justify-center text-white mb-4 group-hover:scale-110 transition-transform duration-300`}>
                  {section.icon}
                </div>
                
                <CardTitle 
                  className="text-xl font-bold text-gray-900 dark:text-white group-hover:text-primary transition-colors duration-300"
                  data-testid={`title-section-${section.id}`}
                >
                  {section.title}
                </CardTitle>
                
                {section.stats && (
                  <div className="text-sm text-muted-foreground font-medium">
                    {section.stats}
                  </div>
                )}
              </CardHeader>
              
              <CardContent className="px-6 pt-0">
                <CardDescription 
                  className="text-gray-600 dark:text-gray-300 mb-6"
                  data-testid={`description-section-${section.id}`}
                >
                  {section.description}
                </CardDescription>
                
                <Button 
                  variant="default" 
                  className={`w-full bg-gradient-to-r ${section.gradient} hover:opacity-90 text-white font-semibold mb-6`}
                  data-testid={`button-section-${section.id}`}
                >
                  {section.cta?.label || "Get Started"}
                  <ArrowRight className="ml-2 h-4 w-4 group-hover:translate-x-1 transition-transform duration-300" />
                </Button>
              </CardContent>
            </Link>

            {/* What's Included Section */}
            <div className="px-6 pb-6 flex-1">
              <h4 className="text-lg font-semibold text-gray-900 dark:text-white mb-4">
                What's Included:
              </h4>
              <div className="space-y-3">
                {sectionDetails[section.id as keyof typeof sectionDetails]?.items.map((item, index) => (
                  <Link key={index} href={item.href}>
                    <div className="flex items-start gap-3 p-3 rounded-lg hover:bg-gray-50 dark:hover:bg-gray-700 transition-colors duration-200 group cursor-pointer">
                      <div className={`min-w-[10px] w-[10px] h-[10px] rounded-full bg-gradient-to-r ${section.gradient} mt-1.5 flex-shrink-0 group-hover:scale-125 transition-transform duration-200`}></div>
                      <div>
                        <div className="font-medium text-gray-900 dark:text-white text-sm group-hover:text-primary transition-colors duration-200">
                          {item.name}
                        </div>
                        <div className="text-xs text-gray-600 dark:text-gray-400 mt-1">
                          {item.desc}
                        </div>
                      </div>
                    </div>
                  </Link>
                ))}
              </div>
            </div>
          </Card>
        ))}
      </div>

      
      {/* Additional CTA section */}
      <div className="text-center mt-16">
        <div className="bg-gradient-to-r from-primary/10 to-accent/10 rounded-2xl p-8 md:p-12">
          <h3 className="text-2xl md:text-3xl font-bold text-gray-900 dark:text-white mb-4">
            Ready to Build Something Amazing?
          </h3>
          <p className="text-lg text-muted-foreground mb-6 max-w-2xl mx-auto">
            Join thousands of entrepreneurs who have successfully launched their brands with our tools.
          </p>
          <div className="flex flex-col sm:flex-row gap-4 justify-center">
            <Link href="/business-development/brand-name-wizard">
              <Button 
                size="lg" 
                className="bg-primary hover:bg-primary/90 text-white px-8 py-3 text-lg font-semibold"
                data-testid="button-start-journey"
              >
                Start Your Journey
              </Button>
            </Link>
            <Link href="/business-development/brand-kit">
              <Button 
                variant="outline" 
                size="lg" 
                className="border-primary text-primary hover:bg-primary hover:text-primary-foreground px-8 py-3 text-lg font-semibold"
                data-testid="button-explore-tools"
              >
                Explore Our Tools
              </Button>
            </Link>
          </div>
        </div>
      </div>
    </section>
  );
}