import { useState } from "react";
import { Button } from "@/components/ui/button";
import { usePlanStoreV2 } from "@/stores/usePlanStore.v2";
import { useBrandKitStore } from "@/stores/useBrandKitStore";
import { exportFullPlanPDF, exportFullPlanDOCX } from "@/services/export";
import { FileText, Download, Loader2 } from "lucide-react";

export function ExportBar({ coverDataUrl }: { coverDataUrl?: string }) {
  const sections = usePlanStoreV2((s) => s.sections);
  const brand = useBrandKitStore((s) => s.getCurrentKit());
  const [isExporting, setIsExporting] = useState<'pdf' | 'docx' | null>(null);
  
  const title = brand?.name ? `${brand.name} – Business Plan` : "Business Plan";

  async function onPdf() {
    setIsExporting('pdf');
    try {
      const result = await exportFullPlanPDF({
        title,
        subtitle: "Generated by IBrandBiz",
        coverDataUrl,
        sections: sections.map(s => ({ id: s.id, title: s.title, content: s.content })),
        includeTOC: true,
      });
      console.log("Exported PDF:", result);
    } catch (error) {
      console.error("PDF export failed:", error);
    } finally {
      setIsExporting(null);
    }
  }

  async function onDocx() {
    setIsExporting('docx');
    try {
      const result = await exportFullPlanDOCX({
        title,
        subtitle: "Generated by IBrandBiz",
        coverDataUrl,
        sections: sections.map(s => ({ id: s.id, title: s.title, content: s.content })),
        includeTOC: true,
      });
      console.log("Exported DOCX:", result);
    } catch (error) {
      console.error("DOCX export failed:", error);
    } finally {
      setIsExporting(null);
    }
  }

  return (
    <div className="flex gap-2" data-testid="export-bar">
      <Button
        variant="outline"
        size="sm"
        onClick={onPdf}
        disabled={isExporting !== null}
        className="flex items-center gap-2"
        data-testid="export-pdf"
      >
        {isExporting === 'pdf' ? (
          <Loader2 className="w-4 h-4 animate-spin" />
        ) : (
          <FileText className="w-4 h-4" />
        )}
        Export PDF
      </Button>
      <Button
        variant="outline"
        size="sm"
        onClick={onDocx}
        disabled={isExporting !== null}
        className="flex items-center gap-2"
        data-testid="export-docx"
      >
        {isExporting === 'docx' ? (
          <Loader2 className="w-4 h-4 animate-spin" />
        ) : (
          <Download className="w-4 h-4" />
        )}
        Export DOCX
      </Button>
    </div>
  );
}