// src/components/CoverDesigner.tsx
import React, { useEffect, useRef, useState } from "react";
import { COVER_THEMES, type CoverTheme } from "../lib/coverThemes";
import { useBrandKitStore } from "../stores/useBrandKitStore";
import { applyBrandTint } from "../lib/coverTint";
import html2canvas from "html2canvas";

export function CoverDesigner({ onSaveImage }:{ onSaveImage?: (dataUrl:string)=> void }) {
  const currentBrand = useBrandKitStore((s)=> s.getCurrentKit());
  const [theme, setTheme] = useState<CoverTheme>(COVER_THEMES[0]);
  const [title, setTitle] = useState("Business Plan");
  const [subtitle, setSubtitle] = useState("for Your Company");
  const [logoUrl, setLogoUrl] = useState<string>("");
  const [applyBrand, setApplyBrand] = useState(true);
  const frameRef = useRef<HTMLDivElement|null>(null);

  // Auto-hydrate from Brand Kit
  useEffect(()=>{
    if (!currentBrand) return;
    if (title === "Business Plan" && currentBrand.name) {
      setTitle(`${currentBrand.name} – Business Plan`);
    }
    if (currentBrand.logoUrl && !logoUrl) setLogoUrl(currentBrand.logoUrl);
  // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [currentBrand?.id]);

  async function exportPng(){
    if (!frameRef.current) return;
    const canvas = await html2canvas(frameRef.current, { backgroundColor: "#ffffff", scale: 2 });
    const dataUrl = canvas.toDataURL("image/png");
    onSaveImage?.(dataUrl);
  }

  return (
    <div className="grid gap-4">
      <div className="grid md:grid-cols-3 gap-3">
        <label className="grid gap-1">
          <span className="text-sm text-gray-600">Theme</span>
          <select className="rounded-xl border p-2" value={theme.id}
                  onChange={(e)=> setTheme(COVER_THEMES.find(t=> t.id===e.target.value) || COVER_THEMES[0])}>
            {COVER_THEMES.map(t=> <option key={t.id} value={t.id}>{t.name}</option>)}
          </select>
        </label>
        <label className="grid gap-1">
          <span className="text-sm text-gray-600">Title</span>
          <input className="rounded-xl border p-2" value={title} onChange={(e)=> setTitle(e.target.value)} />
        </label>
        <label className="grid gap-1">
          <span className="text-sm text-gray-600">Subtitle</span>
          <input className="rounded-xl border p-2" value={subtitle} onChange={(e)=> setSubtitle(e.target.value)} />
        </label>
        <label className="grid gap-1 md:col-span-2">
          <span className="text-sm text-gray-600">Logo URL (optional)</span>
          <input className="rounded-xl border p-2" placeholder="https://..."
                 value={logoUrl} onChange={(e)=> setLogoUrl(e.target.value)} />
        </label>
        <label className="flex items-center gap-2 text-sm">
          <input type="checkbox" checked={applyBrand} onChange={(e)=> setApplyBrand(e.target.checked)} />
          Apply Brand Kit
        </label>
      </div>

      {/* Preview (A4-ish ratio) */}
      <div className="rounded-2xl border p-3">
        <h5 className="font-medium mb-2">Cover Preview</h5>
        <div ref={frameRef} className="relative w-full" style={{ aspectRatio: "1.4142 / 1" }}>
          <CoverCanvas
            theme={theme}
            title={title}
            subtitle={subtitle}
            logoUrl={logoUrl}
            applyBrand={applyBrand}
            brandPrimary={currentBrand?.primary}
            brandSecondary={currentBrand?.secondary}
            fontHead={currentBrand?.fontHead}
            fontBody={currentBrand?.fontBody}
          />
        </div>
        <div className="mt-2 flex justify-end">
          <button className="rounded-2xl border px-3 py-2" onClick={exportPng}>Export Cover PNG</button>
        </div>
      </div>
    </div>
  );
}

function CoverCanvas({
  theme, title, subtitle, logoUrl, applyBrand, brandPrimary, brandSecondary, fontHead, fontBody
}: {
  theme: CoverTheme;
  title: string;
  subtitle: string;
  logoUrl?: string;
  applyBrand?: boolean;
  brandPrimary?: string;
  brandSecondary?: string;
  fontHead?: string;
  fontBody?: string;
}) {
  // base colors from theme
  let primary = theme.primary;
  let secondary = theme.secondary || theme.primary;

  // blend in brand tint if requested
  if (applyBrand && brandPrimary) {
    primary = applyBrandTint(primary, brandPrimary, 0.18);
    if (brandSecondary) secondary = applyBrandTint(secondary, brandSecondary, 0.12);
  }

  // background variants
  const grad = theme.style === "Gradient"
    ? { backgroundImage: `linear-gradient(135deg, ${primary}, ${secondary})` }
    : {};
  const bg   = theme.style === "Minimal"
    ? { backgroundColor: primary }
    : {};
  const patternStyle: React.CSSProperties =
    theme.pattern === "dots"
      ? { backgroundImage: "radial-gradient(currentColor 1px, transparent 1px)", backgroundSize: "16px 16px", color: "rgba(255,255,255,0.15)" }
      : theme.pattern === "grid"
      ? { backgroundImage: "linear-gradient(rgba(255,255,255,0.12) 1px, transparent 1px), linear-gradient(90deg, rgba(255,255,255,0.12) 1px, transparent 1px)", backgroundSize: "20px 20px" }
      : theme.pattern === "waves"
      ? { backgroundImage: "radial-gradient(ellipse at 20% 20%, rgba(255,255,255,0.12), transparent 35%), radial-gradient(ellipse at 80% 80%, rgba(255,255,255,0.12), transparent 35%)" }
      : {};

  const headFont = fontHead || "Inter, ui-sans-serif";
  const bodyFont = fontBody || "Inter, ui-sans-serif";

  return (
    <div className="w-full h-full rounded-xl overflow-hidden text-white p-10 flex flex-col justify-between"
         style={{ ...grad, ...bg, fontFamily: bodyFont }}>
      <div className="absolute inset-0 pointer-events-none" style={patternStyle} />
      <div className="flex items-center justify-between relative">
        <div className="text-sm opacity-80" style={{ fontFamily: bodyFont }}>IBrandBiz</div>
        {logoUrl ? <img src={logoUrl} alt="logo" className="h-10 w-auto object-contain" /> : <div />}
      </div>

      <div className="relative" style={{ fontFamily: headFont }}>
        <h1 className="text-4xl md:text-6xl font-bold leading-tight">{title}</h1>
        <p className="text-lg md:text-2xl mt-3 opacity-90">{subtitle}</p>
      </div>

      <div className="flex items-center justify-between relative" style={{ fontFamily: bodyFont }}>
        <div className="text-sm opacity-80">{new Date().toLocaleDateString()}</div>
        <div className="text-sm opacity-80">www.ibrandbiz.com</div>
      </div>
    </div>
  );
}