import { useState, useEffect } from "react";
import { Link } from "wouter";
import { Button } from "@/components/ui/button";
import { Switch } from "@/components/ui/switch";
import { Card } from "@/components/ui/card";
import { getConsent, setConsent, type ConsentState } from "@/lib/consent";
import { X, Cookie } from "lucide-react";

/**
 * CookieConsentBanner Component
 * Based on SuperNova's specification for IBrandBiz
 * 
 * Features:
 * - Fixed bottom banner, shows only when no consent exists
 * - Three category toggles: Essential (always on), Analytics, Marketing
 * - Three action buttons: Reject non-essential, Save choices, Accept all
 * - Links to /cookie-policy page
 * - Responsive design with IBrandBiz branding
 */
export function CookieConsentBanner() {
  const [showBanner, setShowBanner] = useState(false);
  const [preferences, setPreferences] = useState<ConsentState>({
    essential: true,
    analytics: true,
    marketing: false,
  });

  // Check if consent exists on mount
  useEffect(() => {
    const existingConsent = getConsent();
    if (!existingConsent) {
      setShowBanner(true);
    }
  }, []);

  // Handle consent decisions
  const handleConsent = (consentData: ConsentState) => {
    setConsent(consentData);
    setShowBanner(false);

    // TODO: Initialize analytics scripts based on consent
    if (consentData.analytics) {
      // TODO: Initialize Google Analytics, tracking pixels, etc.
      console.log("Analytics consent granted - initialize tracking scripts");
    }

    // TODO: Initialize marketing scripts based on consent  
    if (consentData.marketing) {
      // TODO: Initialize marketing pixels, retargeting scripts, etc.
      console.log("Marketing consent granted - initialize marketing scripts");
    }
  };

  const handleRejectNonEssential = () => {
    handleConsent({
      essential: true,
      analytics: false,
      marketing: false,
    });
  };

  const handleSaveChoices = () => {
    handleConsent(preferences);
  };

  const handleAcceptAll = () => {
    handleConsent({
      essential: true,
      analytics: true,
      marketing: true,
    });
  };

  if (!showBanner) {
    return null;
  }

  return (
    <div className="fixed bottom-0 left-0 right-0 z-50 p-4 bg-black/50 backdrop-blur-sm">
      <Card className="mx-auto max-w-4xl bg-white dark:bg-ibrand-navy border-ibrand-medium shadow-xl">
        <div className="p-6">
          <div className="flex items-start gap-4">
            {/* Cookie Icon */}
            <div className="flex-shrink-0 p-2 bg-ibrand-pale-blue dark:bg-ibrand-dark-blue rounded-lg">
              <Cookie className="h-6 w-6 text-ibrand-medium-blue" />
            </div>

            {/* Content */}
            <div className="flex-1 space-y-4">
              {/* Header */}
              <div>
                <h3 className="text-lg font-semibold text-ibrand-navy dark:text-ibrand-white">
                  Cookie Preferences
                </h3>
                <p className="text-sm text-ibrand-text-muted mt-1">
                  We use cookies to enhance your experience, analyze site usage, and assist in marketing. 
                  You can manage your preferences below.{" "}
                  <Link 
                    href="/cookie-policy" 
                    className="text-ibrand-medium-blue hover:text-ibrand-dark-blue underline"
                    data-testid="link-cookie-policy"
                  >
                    Learn more in our Cookie Policy
                  </Link>
                </p>
              </div>

              {/* Cookie Categories */}
              <div className="grid gap-3 sm:grid-cols-3">
                {/* Essential Cookies */}
                <div className="flex items-center justify-between p-3 rounded-lg bg-ibrand-pale-blue/50 dark:bg-ibrand-dark-blue/30">
                  <div className="flex-1">
                    <label className="text-sm font-medium text-ibrand-navy dark:text-ibrand-white">
                      Essential
                    </label>
                    <p className="text-xs text-ibrand-text-muted">
                      Required for site functionality
                    </p>
                  </div>
                  <Switch
                    checked={true}
                    disabled={true}
                    data-testid="switch-essential"
                    className="opacity-50"
                  />
                </div>

                {/* Analytics Cookies */}
                <div className="flex items-center justify-between p-3 rounded-lg bg-ibrand-pale-blue/30 dark:bg-ibrand-dark-blue/20">
                  <div className="flex-1">
                    <label className="text-sm font-medium text-ibrand-navy dark:text-ibrand-white">
                      Analytics
                    </label>
                    <p className="text-xs text-ibrand-text-muted">
                      Help us improve our site
                    </p>
                  </div>
                  <Switch
                    checked={preferences.analytics}
                    onCheckedChange={(checked) =>
                      setPreferences(prev => ({ ...prev, analytics: checked }))
                    }
                    data-testid="switch-analytics"
                  />
                </div>

                {/* Marketing Cookies */}
                <div className="flex items-center justify-between p-3 rounded-lg bg-ibrand-pale-blue/30 dark:bg-ibrand-dark-blue/20">
                  <div className="flex-1">
                    <label className="text-sm font-medium text-ibrand-navy dark:text-ibrand-white">
                      Marketing
                    </label>
                    <p className="text-xs text-ibrand-text-muted">
                      Personalized ads & content
                    </p>
                  </div>
                  <Switch
                    checked={preferences.marketing}
                    onCheckedChange={(checked) =>
                      setPreferences(prev => ({ ...prev, marketing: checked }))
                    }
                    data-testid="switch-marketing"
                  />
                </div>
              </div>

              {/* Action Buttons */}
              <div className="flex flex-col sm:flex-row gap-3 pt-2">
                <Button
                  variant="outline"
                  onClick={handleRejectNonEssential}
                  className="border-ibrand-medium hover:bg-ibrand-pale-blue hover:border-ibrand-dark-blue"
                  data-testid="button-reject-non-essential"
                >
                  Reject non-essential
                </Button>
                
                <Button
                  variant="secondary"
                  onClick={handleSaveChoices}
                  className="bg-ibrand-secondary hover:bg-ibrand-secondary-hover text-white"
                  data-testid="button-save-choices"
                >
                  Save choices
                </Button>
                
                <Button
                  onClick={handleAcceptAll}
                  className="bg-ibrand-primary hover:bg-ibrand-primary-hover text-white"
                  data-testid="button-accept-all"
                >
                  Accept all
                </Button>
              </div>
            </div>

            {/* Close Button */}
            <Button
              variant="ghost"
              size="icon"
              onClick={() => setShowBanner(false)}
              className="flex-shrink-0 h-8 w-8 text-ibrand-text-muted hover:text-ibrand-navy dark:hover:text-ibrand-white"
              data-testid="button-close-banner"
            >
              <X className="h-4 w-4" />
              <span className="sr-only">Close banner</span>
            </Button>
          </div>
        </div>
      </Card>
    </div>
  );
}