import { useRef, useState } from "react";
import { captureElementToPng, copyDataUrlToClipboard } from "@/lib/snapshot";
import { usePlanStoreV2 } from "@/stores/usePlanStore.v2";
import { Check, Copy, Camera, AlertCircle } from "lucide-react";

interface ChartSnapshotButtonV2Props {
  targetSectionId: string;
  getElement: () => HTMLElement | null;
  title?: string;
  className?: string;
}

export function ChartSnapshotButtonV2({
  targetSectionId,
  getElement,
  title = "Chart Snapshot",
  className = "",
}: ChartSnapshotButtonV2Props) {
  const update = usePlanStoreV2((s) => s.updateSection);
  const [insertState, setInsertState] = useState<"idle" | "busy" | "success" | "error">("idle");
  const [copyState, setCopyState] = useState<"idle" | "busy" | "success" | "error">("idle");
  const busy = useRef(false);

  const resetStates = () => {
    setTimeout(() => {
      setInsertState("idle");
      setCopyState("idle");
    }, 2000);
  };

  async function insertSnapshot() {
    if (busy.current) return;
    busy.current = true;
    setInsertState("busy");
    
    try {
      const el = getElement();
      if (!el) {
        throw new Error("Element not found");
      }
      
      const dataUrl = await captureElementToPng(el, { scale: 2 });
      const md = `### ${title}

![${title}](${dataUrl})`;
      
      update(targetSectionId, (prev: any) => ({
        content: `${prev?.content ? prev.content + "\n\n" : ""}${md}`,
      }));
      
      setInsertState("success");
      resetStates();
    } catch (error) {
      console.error("Failed to insert snapshot:", error);
      setInsertState("error");
      resetStates();
    } finally {
      busy.current = false;
    }
  }

  async function copyToPng() {
    if (busy.current) return;
    busy.current = true;
    setCopyState("busy");
    
    try {
      const el = getElement();
      if (!el) {
        throw new Error("Element not found");
      }
      
      const dataUrl = await captureElementToPng(el, { scale: 2 });
      await copyDataUrlToClipboard(dataUrl);
      
      setCopyState("success");
      resetStates();
    } catch (error) {
      console.error("Failed to copy to clipboard:", error);
      setCopyState("error");
      resetStates();
    } finally {
      busy.current = false;
    }
  }

  const getButtonContent = (state: string, defaultText: string, icon: any) => {
    switch (state) {
      case "busy":
        return (
          <>
            <div className="w-4 h-4 border-2 border-current border-t-transparent rounded-full animate-spin" />
            Processing...
          </>
        );
      case "success":
        return (
          <>
            <Check className="w-4 h-4" />
            Success!
          </>
        );
      case "error":
        return (
          <>
            <AlertCircle className="w-4 h-4" />
            Failed
          </>
        );
      default:
        return (
          <>
            {icon}
            {defaultText}
          </>
        );
    }
  };

  const getButtonClass = (state: string) => {
    const base = "flex items-center gap-2 rounded-md border px-3 py-2 text-sm transition-colors disabled:opacity-50";
    switch (state) {
      case "success":
        return `${base} bg-green-50 border-green-200 text-green-700 hover:bg-green-100`;
      case "error":
        return `${base} bg-red-50 border-red-200 text-red-700 hover:bg-red-100`;
      default:
        return `${base} hover:bg-gray-50`;
    }
  };

  return (
    <div className={`flex items-center gap-2 ${className}`}>
      <button
        className={getButtonClass(insertState)}
        onClick={insertSnapshot}
        disabled={busy.current}
        data-testid="button-snapshot-insert"
      >
        {getButtonContent(insertState, "Insert Snapshot (PNG)", <Camera className="w-4 h-4" />)}
      </button>
      
      <button
        className={getButtonClass(copyState)}
        onClick={copyToPng}
        disabled={busy.current}
        data-testid="button-snapshot-copy"
      >
        {getButtonContent(copyState, "Copy PNG", <Copy className="w-4 h-4" />)}
      </button>
    </div>
  );
}