import { useState } from 'react';
import { ShoppingCart, X, Plus, Minus, Trash2, CreditCard, AlertCircle } from 'lucide-react';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { Separator } from '@/components/ui/separator';
import { useCart } from '@/contexts/CartContext';
import { useAuth } from '@/contexts/AuthContext';
import { useMutation } from '@tanstack/react-query';
import { apiRequest } from '@/lib/queryClient';
import { useToast } from '@/hooks/use-toast';
import {
  Sheet,
  SheetContent,
  SheetDescription,
  SheetHeader,
  SheetTitle,
  SheetTrigger,
} from '@/components/ui/sheet';

interface CartDrawerProps {
  trigger?: React.ReactNode;
}

export function CartDrawer({ trigger }: CartDrawerProps) {
  const [open, setOpen] = useState(false);
  
  // Safely handle cart context - if CartProvider is not available, don't render cart functionality
  let cartContext;
  try {
    cartContext = useCart();
  } catch (error) {
    // CartProvider not available, render disabled cart button
    const disabledTrigger = (
      <Button
        variant="outline"
        size="sm"
        className="relative opacity-50 cursor-not-allowed"
        data-testid="button-cart-disabled"
        disabled
      >
        <ShoppingCart className="h-4 w-4" />
      </Button>
    );
    return trigger || disabledTrigger;
  }

  const { items, itemCount, totalCents, removeItem, updateQuantity, clearCart, sessionId } = cartContext;
  const { user } = useAuth();
  const { toast } = useToast();

  const checkoutMutation = useMutation({
    mutationFn: async () => {
      // Determine the correct checkout endpoint based on cart contents
      const hasCreatorAssets = items.some(item => item.itemType === 'creator_asset');
      const hasOtherItems = items.some(item => item.itemType !== 'creator_asset');
      
      // This should never happen due to cart validation, but double-check
      if (hasCreatorAssets && hasOtherItems) {
        throw new Error('Cannot checkout with mixed item types. Please remove conflicting items.');
      }
      
      // Route to appropriate checkout endpoint
      const endpoint = hasCreatorAssets 
        ? '/api/checkout/creator-marketplace'
        : '/api/checkout/create-session';
      
      return apiRequest(endpoint, {
        method: 'POST',
        body: { 
          cartItems: items,
          sessionId: !user ? sessionId : undefined
        }
      });
    },
    onSuccess: (data) => {
      // Redirect to Stripe checkout
      window.location.href = data.url;
    },
    onError: (error: any) => {
      toast({
        title: "Checkout failed",
        description: error.message || "Failed to create checkout session",
        variant: "destructive",
      });
    }
  });

  const handleCheckout = () => {
    if (items.length === 0) {
      toast({
        title: "Cart is empty",
        description: "Add items to your cart before checking out",
        variant: "destructive",
      });
      return;
    }

    // Validate that cart doesn't mix incompatible item types
    const hasSubscription = items.some(item => item.itemType === 'subscription');
    const hasStockPhoto = items.some(item => item.itemType === 'stock_photo');
    const hasCreatorAsset = items.some(item => item.itemType === 'creator_asset');
    
    if (hasSubscription && hasStockPhoto) {
      toast({
        title: "Mixed cart not allowed",
        description: "Please purchase subscription plans and stock photos separately",
        variant: "destructive",
      });
      return;
    }
    
    if (hasCreatorAsset && (hasSubscription || hasStockPhoto)) {
      toast({
        title: "Mixed cart not allowed",
        description: "Creator assets must be purchased separately from other item types",
        variant: "destructive",
      });
      return;
    }

    // CRITICAL: Defensive validation for creator assets - ensure single seller constraint
    if (hasCreatorAsset) {
      const creatorAssets = items.filter(item => item.itemType === 'creator_asset');
      
      // Validate all creator assets share the same creatorId
      const creatorIds = [...new Set(creatorAssets.map(item => item.metadata?.creatorId).filter(Boolean))];
      if (creatorIds.length > 1) {
        toast({
          title: "Mixed creator cart detected",
          description: "All creator assets must be from the same creator. Please remove items from other creators.",
          variant: "destructive",
        });
        return;
      }
      
      // Validate no creator asset has quantity > 1 (defensive check)
      const invalidQuantityAsset = creatorAssets.find(item => item.quantity > 1);
      if (invalidQuantityAsset) {
        toast({
          title: "Invalid quantity detected",
          description: "Digital licenses are limited to one copy per asset. Please refresh your cart.",
          variant: "destructive",
        });
        return;
      }
    }

    checkoutMutation.mutate();
  };

  const formatPrice = (cents: number) => {
    return `$${(cents / 100).toFixed(2)}`;
  };

  const getItemTypeLabel = (type: string) => {
    switch (type) {
      case 'stock_photo': return 'Stock Photo';
      case 'subscription': return 'Subscription';
      case 'creator_asset': return 'Creator Asset';
      default: return 'Unknown';
    }
  };

  const getItemTypeBadgeVariant = (type: string): "default" | "secondary" | "outline" => {
    switch (type) {
      case 'stock_photo': return 'secondary';
      case 'subscription': return 'default';
      case 'creator_asset': return 'outline';
      default: return 'secondary';
    }
  };

  const defaultTrigger = (
    <Button
      variant="outline"
      size="sm"
      className="relative"
      data-testid="button-cart-open"
    >
      <ShoppingCart className="h-4 w-4" />
      {itemCount > 0 && (
        <Badge 
          variant="destructive" 
          className="absolute -top-2 -right-2 h-5 w-5 rounded-full p-0 flex items-center justify-center text-xs"
          data-testid="badge-cart-count"
        >
          {itemCount}
        </Badge>
      )}
    </Button>
  );

  return (
    <Sheet open={open} onOpenChange={setOpen}>
      <SheetTrigger asChild>
        {trigger || defaultTrigger}
      </SheetTrigger>
      <SheetContent className="w-[400px] sm:w-[540px]" data-testid="drawer-cart">
        <SheetHeader>
          <SheetTitle className="flex items-center gap-2" data-testid="text-cart-title">
            <ShoppingCart className="h-5 w-5" />
            Shopping Cart ({itemCount} items)
          </SheetTitle>
          <SheetDescription data-testid="text-cart-description">
            Review your items before checkout
          </SheetDescription>
        </SheetHeader>

        <div className="flex flex-col h-full pt-6">
          {items.length === 0 ? (
            <div className="flex flex-col items-center justify-center flex-1 text-center" data-testid="cart-empty-state">
              <ShoppingCart className="h-16 w-16 text-gray-300 mb-4" />
              <p className="text-lg font-medium text-gray-600 mb-2">Your cart is empty</p>
              <p className="text-sm text-gray-500">Add some items to get started</p>
            </div>
          ) : (
            <>
              {/* Cart Items */}
              <div className="flex-1 overflow-y-auto space-y-4 pr-2">
                {items.map((item) => (
                  <div key={item.id} className="flex items-center space-x-3 p-3 bg-gray-50 rounded-lg" data-testid={`cart-item-${item.id}`}>
                    <div className="flex-1 min-w-0">
                      <div className="flex items-center gap-2 mb-1">
                        <h4 className="text-sm font-medium truncate" data-testid={`text-item-name-${item.id}`}>
                          {item.itemName}
                        </h4>
                        <Badge 
                          variant={getItemTypeBadgeVariant(item.itemType)}
                          className="text-xs"
                          data-testid={`badge-item-type-${item.id}`}
                        >
                          {getItemTypeLabel(item.itemType)}
                        </Badge>
                      </div>
                      {/* Show creator name for creator assets */}
                      {item.itemType === 'creator_asset' && item.metadata?.creatorName && (
                        <p className="text-xs text-gray-500 mb-1" data-testid={`text-creator-name-${item.id}`}>
                          by {item.metadata.creatorName}
                        </p>
                      )}
                      <p className="text-sm text-gray-600" data-testid={`text-item-price-${item.id}`}>
                        {formatPrice(item.itemPrice)}
                        {item.itemType === 'creator_asset' && (
                          <span className="text-xs text-gray-500 ml-1">(Digital License)</span>
                        )}
                      </p>
                    </div>
                    
                    {/* Quantity Controls */}
                    <div className="flex items-center space-x-1">
                      {item.itemType === 'creator_asset' ? (
                        // For creator assets, show fixed quantity with disabled controls
                        <div className="flex items-center space-x-1 opacity-50">
                          <Button
                            variant="outline"
                            size="sm"
                            className="h-8 w-8 p-0"
                            disabled
                            data-testid={`button-quantity-decrease-${item.id}`}
                          >
                            <Minus className="h-3 w-3" />
                          </Button>
                          <span className="w-8 text-center text-sm font-medium" data-testid={`text-quantity-${item.id}`}>
                            {item.quantity}
                          </span>
                          <Button
                            variant="outline"
                            size="sm"
                            className="h-8 w-8 p-0"
                            disabled
                            data-testid={`button-quantity-increase-${item.id}`}
                          >
                            <Plus className="h-3 w-3" />
                          </Button>
                        </div>
                      ) : (
                        // For other item types, show normal quantity controls
                        <>
                          <Button
                            variant="outline"
                            size="sm"
                            className="h-8 w-8 p-0"
                            onClick={() => updateQuantity(item.id, item.quantity - 1)}
                            data-testid={`button-quantity-decrease-${item.id}`}
                          >
                            <Minus className="h-3 w-3" />
                          </Button>
                          <span className="w-8 text-center text-sm font-medium" data-testid={`text-quantity-${item.id}`}>
                            {item.quantity}
                          </span>
                          <Button
                            variant="outline"
                            size="sm"
                            className="h-8 w-8 p-0"
                            onClick={() => updateQuantity(item.id, item.quantity + 1)}
                            data-testid={`button-quantity-increase-${item.id}`}
                          >
                            <Plus className="h-3 w-3" />
                          </Button>
                        </>
                      )}
                    </div>

                    {/* Remove Button */}
                    <Button
                      variant="outline"
                      size="sm"
                      className="h-8 w-8 p-0 text-red-500 hover:text-red-700 hover:bg-red-50"
                      onClick={() => removeItem(item.id)}
                      data-testid={`button-remove-item-${item.id}`}
                    >
                      <Trash2 className="h-3 w-3" />
                    </Button>
                  </div>
                ))}
              </div>

              <Separator className="my-4" />

              {/* Cart Summary */}
              <div className="space-y-4">
                {/* Validation Warning */}
                {(() => {
                  const hasSubscription = items.some(item => item.itemType === 'subscription');
                  const hasStockPhoto = items.some(item => item.itemType === 'stock_photo');
                  const hasCreatorAsset = items.some(item => item.itemType === 'creator_asset');
                  
                  if (hasSubscription && hasStockPhoto) {
                    return (
                      <div className="flex items-start gap-2 p-3 bg-yellow-50 border border-yellow-200 rounded-lg" data-testid="cart-validation-warning">
                        <AlertCircle className="h-4 w-4 text-yellow-600 mt-0.5" />
                        <div className="text-sm text-yellow-800">
                          <strong>Mixed cart detected:</strong> Subscription plans and stock photos must be purchased separately. Please remove items of one type before checkout.
                        </div>
                      </div>
                    );
                  }
                  
                  if (hasCreatorAsset && (hasSubscription || hasStockPhoto)) {
                    return (
                      <div className="flex items-start gap-2 p-3 bg-yellow-50 border border-yellow-200 rounded-lg" data-testid="cart-validation-warning">
                        <AlertCircle className="h-4 w-4 text-yellow-600 mt-0.5" />
                        <div className="text-sm text-yellow-800">
                          <strong>Mixed cart detected:</strong> Creator assets must be purchased separately from other item types. Please remove conflicting items before checkout.
                        </div>
                      </div>
                    );
                  }
                  
                  return null;
                })()}

                {/* Total */}
                <div className="flex justify-between items-center">
                  <span className="text-lg font-semibold" data-testid="text-cart-total-label">Total:</span>
                  <span className="text-lg font-bold" data-testid="text-cart-total">
                    {formatPrice(totalCents)}
                  </span>
                </div>

                {/* Action Buttons */}
                <div className="flex space-x-2">
                  <Button
                    variant="outline"
                    onClick={clearCart}
                    className="flex-1"
                    data-testid="button-clear-cart"
                  >
                    Clear Cart
                  </Button>
                  <Button
                    onClick={handleCheckout}
                    disabled={checkoutMutation.isPending || items.length === 0}
                    className="flex-1"
                    data-testid="button-checkout"
                  >
                    {checkoutMutation.isPending ? (
                      "Creating checkout..."
                    ) : (
                      <>
                        <CreditCard className="h-4 w-4 mr-2" />
                        Checkout
                      </>
                    )}
                  </Button>
                </div>

                {/* Auth Status */}
                {!user && (
                  <p className="text-xs text-gray-500 text-center" data-testid="text-guest-notice">
                    You're shopping as a guest. Sign in to save your cart across devices.
                  </p>
                )}
              </div>
            </>
          )}
        </div>
      </SheetContent>
    </Sheet>
  );
}