import { useState } from "react";
import { useMutation } from "@tanstack/react-query";
import { useLocation } from "wouter";
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogFooter,
  DialogHeader,
  DialogTitle,
} from "@/components/ui/dialog";
import {
  Form,
  FormControl,
  FormField,
  FormItem,
  FormLabel,
  FormMessage,
} from "@/components/ui/form";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { Button } from "@/components/ui/button";
import { Textarea } from "@/components/ui/textarea";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { Heart, Loader2, Pause } from "lucide-react";
import { toast } from "@/hooks/use-toast";
import { apiRequest } from "@/lib/queryClient";
import { insertCancellationSchema, CANCELLATION_REASONS, type InsertCancellation } from "@shared/schema";

interface CancelSurveyProps {
  isOpen: boolean;
  onClose: () => void;
  onKeepPlan: () => void;
  userEmail?: string;
}

// Use shared schema for validation consistency
type CancelSurveyFormData = Omit<InsertCancellation, 'userId' | 'firebaseUid' | 'email'>;

const reasonOptions = [
  { value: CANCELLATION_REASONS.TOO_EXPENSIVE, label: "Too expensive" },
  { value: CANCELLATION_REASONS.DIDNT_GET_VALUE, label: "Didn't get the value I expected" },
  { value: CANCELLATION_REASONS.MISSING_FEATURE, label: "Missing features I need" },
  { value: CANCELLATION_REASONS.TECHNICAL_ISSUES, label: "Technical issues or bugs" },
  { value: CANCELLATION_REASONS.TEMPORARY_PAUSE, label: "Taking a temporary break" },
  { value: CANCELLATION_REASONS.OTHER, label: "Other reason" },
];

export function CancelSurvey({ 
  isOpen, 
  onClose, 
  onKeepPlan,
  userEmail 
}: CancelSurveyProps) {
  const [isSubmitting, setIsSubmitting] = useState(false);
  const [isPausing, setIsPausing] = useState(false);
  const [, setLocation] = useLocation();

  // Use shared schema for consistent validation
  const cancelSurveySchema = insertCancellationSchema.pick({ reason: true, note: true });
  
  const form = useForm<CancelSurveyFormData>({
    resolver: zodResolver(cancelSurveySchema),
    defaultValues: {
      reason: undefined,
      note: "",
    },
  });

  const cancelMutation = useMutation({
    mutationFn: async (data: CancelSurveyFormData) => {
      return apiRequest("/api/billing/cancel", {
        method: "POST",
        body: JSON.stringify(data),
      });
    },
    onSuccess: () => {
      toast({
        title: "Subscription cancelled",
        description: "Your subscription has been cancelled. We're sorry to see you go!",
      });
      onClose();
      // Navigate to pricing page after a short delay using wouter
      setTimeout(() => {
        setLocation('/pricing');
      }, 2000);
    },
    onError: (error) => {
      console.error("Cancellation error:", error);
      toast({
        variant: "destructive",
        title: "Error",
        description: "Failed to cancel subscription. Please try again or contact support.",
      });
    },
  });

  const pauseMutation = useMutation({
    mutationFn: async (data: { months: number; reason?: string; note?: string }) => {
      return apiRequest("/api/billing/pause", {
        method: "POST",
        body: JSON.stringify(data),
      });
    },
    onSuccess: (_, variables) => {
      toast({
        title: "Subscription paused",
        description: `Your subscription has been paused for ${variables.months} month${variables.months > 1 ? 's' : ''}. You can reactivate anytime!`,
      });
      onClose();
    },
    onError: (error) => {
      console.error("Pause error:", error);
      toast({
        variant: "destructive",
        title: "Error",
        description: "Failed to pause subscription. Please try again or contact support.",
      });
    },
  });

  const onSubmit = async (data: CancelSurveyFormData) => {
    setIsSubmitting(true);
    try {
      await cancelMutation.mutateAsync(data);
    } finally {
      setIsSubmitting(false);
    }
  };

  const handlePause = async (months: number) => {
    setIsPausing(true);
    try {
      const formData = form.getValues();
      await pauseMutation.mutateAsync({
        months,
        reason: formData.reason,
        note: formData.note,
      });
    } finally {
      setIsPausing(false);
    }
  };

  const handleKeepPlan = () => {
    form.reset();
    onKeepPlan();
  };

  const handleClose = () => {
    form.reset();
    onClose();
  };

  return (
    <Dialog open={isOpen} onOpenChange={handleClose}>
      <DialogContent className="sm:max-w-[500px]" data-testid="cancel-survey-modal">
        <DialogHeader>
          <div className="flex items-center gap-2">
            <Heart className="h-6 w-6 text-red-500" />
            <DialogTitle data-testid="cancel-survey-title">
              Before you go...
            </DialogTitle>
          </div>
          <DialogDescription data-testid="cancel-survey-description">
            We're sorry to see you leave! Your feedback helps us improve our service. 
            Could you tell us why you're cancelling?
          </DialogDescription>
        </DialogHeader>

        <Form {...form}>
          <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-6">
            <FormField
              control={form.control}
              name="reason"
              render={({ field }) => (
                <FormItem>
                  <FormLabel>Reason for cancelling *</FormLabel>
                  <Select 
                    onValueChange={field.onChange} 
                    defaultValue={field.value}
                    data-testid="select-cancel-reason"
                  >
                    <FormControl>
                      <SelectTrigger>
                        <SelectValue placeholder="Please select a reason" />
                      </SelectTrigger>
                    </FormControl>
                    <SelectContent>
                      {reasonOptions.map((option) => (
                        <SelectItem 
                          key={option.value} 
                          value={option.value}
                          data-testid={`option-${option.value.toLowerCase().replace(/\s+/g, '-')}`}
                        >
                          {option.label}
                        </SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                  <FormMessage />
                </FormItem>
              )}
            />

            <FormField
              control={form.control}
              name="note"
              render={({ field }) => (
                <FormItem>
                  <FormLabel>Additional comments (optional)</FormLabel>
                  <FormControl>
                    <Textarea
                      placeholder="Is there anything else you'd like us to know? We appreciate your feedback."
                      className="min-h-[100px]"
                      data-testid="textarea-cancel-note"
                      {...field}
                    />
                  </FormControl>
                  <FormMessage />
                </FormItem>
              )}
            />

            {/* Pause options section */}
            <div className="space-y-4 pt-2 border-t">
              <div className="text-center">
                <p className="text-sm text-muted-foreground mb-3" data-testid="text-pause-suggestion">
                  <Pause className="inline w-4 h-4 mr-1" />
                  Prefer to pause instead?
                </p>
                <div className="flex gap-2 justify-center">
                  <Button
                    type="button"
                    variant="secondary"
                    size="sm"
                    onClick={() => handlePause(1)}
                    disabled={isSubmitting || isPausing}
                    data-testid="button-pause-1-month"
                  >
                    {isPausing ? (
                      <>
                        <Loader2 className="w-4 h-4 mr-2 animate-spin" />
                        Pausing...
                      </>
                    ) : (
                      "Pause 1 month"
                    )}
                  </Button>
                  <Button
                    type="button"
                    variant="secondary"
                    size="sm"
                    onClick={() => handlePause(2)}
                    disabled={isSubmitting || isPausing}
                    data-testid="button-pause-2-months"
                  >
                    {isPausing ? (
                      <>
                        <Loader2 className="w-4 h-4 mr-2 animate-spin" />
                        Pausing...
                      </>
                    ) : (
                      "Pause 2 months"
                    )}
                  </Button>
                </div>
              </div>
            </div>

            <DialogFooter className="flex gap-3 sm:gap-3">
              <Button
                type="button"
                variant="outline"
                onClick={handleKeepPlan}
                disabled={isSubmitting || isPausing}
                data-testid="button-keep-plan"
              >
                Keep my plan
              </Button>
              <Button
                type="submit"
                variant="destructive"
                disabled={isSubmitting || isPausing}
                data-testid="button-confirm-cancel"
              >
                {isSubmitting ? (
                  <>
                    <Loader2 className="w-4 h-4 mr-2 animate-spin" />
                    Cancelling...
                  </>
                ) : (
                  "Confirm cancellation"
                )}
              </Button>
            </DialogFooter>
          </form>
        </Form>
      </DialogContent>
    </Dialog>
  );
}