import { useState, useEffect } from "react";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Badge } from "@/components/ui/badge";
import { Separator } from "@/components/ui/separator";
import { useBrandKitStore } from "@/stores/useBrandKitStore";
import { getBrandPalette, applyBrandKitStyles } from "@/lib/brand";
import { Palette, RotateCcw, Image, Type, Sparkles } from "lucide-react";
import { useToast } from "@/hooks/use-toast";

export function BrandKitPanel() {
  const { current: brand, updateKit } = useBrandKitStore();
  const { toast } = useToast();
  const [previewPalette, setPreviewPalette] = useState<string[]>([]);

  // Update palette preview when brand changes
  useEffect(() => {
    const palette = getBrandPalette(brand);
    setPreviewPalette(palette);
    applyBrandKitStyles(brand);
  }, [brand]);

  const handleInputChange = (field: keyof typeof brand) => (
    e: React.ChangeEvent<HTMLInputElement>
  ) => {
    updateKit(brand.id, { [field]: e.target.value });
  };

  const handleReset = () => {
    useBrandKitStore.getState().setDefault("default");
    toast({
      title: "Brand Kit Reset",
      description: "Brand kit has been reset to default IBrandBiz colors.",
    });
  };

  const handleQuickPreset = (preset: "corporate" | "modern" | "vibrant") => {
    const presets = {
      corporate: {
        primary: "#1f2937", // Gray-800
        secondary: "#3b82f6", // Blue-500
        accent: "#10b981", // Emerald-500
        fontHead: "Georgia, serif",
        fontBody: "ui-sans-serif, system-ui",
      },
      modern: {
        primary: "#6366f1", // Indigo-500
        secondary: "#8b5cf6", // Violet-500
        accent: "#ec4899", // Pink-500
        fontHead: "ui-sans-serif, system-ui",
        fontBody: "ui-sans-serif, system-ui",
      },
      vibrant: {
        primary: "#dc2626", // Red-600
        secondary: "#ea580c", // Orange-600
        accent: "#ca8a04", // Yellow-600
        fontHead: "ui-serif, Georgia",
        fontBody: "ui-sans-serif, system-ui",
      },
    };

    updateKit(brand.id, presets[preset]);
    toast({
      title: "Preset Applied",
      description: `${preset.charAt(0).toUpperCase() + preset.slice(1)} preset has been applied to your brand kit.`,
    });
  };

  return (
    <Card className="w-full" data-testid="brand-kit-panel">
      <CardHeader>
        <div className="flex items-center justify-between">
          <CardTitle className="flex items-center gap-2">
            <Palette className="w-5 h-5 text-purple-500" />
            Brand Kit
          </CardTitle>
          <div className="flex items-center gap-2">
            <Badge variant="outline" className="text-xs">
              Auto-applied to charts
            </Badge>
            <Button
              variant="outline"
              size="sm"
              onClick={handleReset}
              data-testid="brand-kit-reset-button"
            >
              <RotateCcw className="w-4 h-4 mr-1" />
              Reset
            </Button>
          </div>
        </div>
      </CardHeader>

      <CardContent className="space-y-6">
        {/* Quick Presets */}
        <div className="space-y-2">
          <Label className="text-sm font-medium flex items-center gap-2">
            <Sparkles className="w-4 h-4 text-amber-500" />
            Quick Presets
          </Label>
          <div className="flex gap-2">
            <Button
              variant="outline"
              size="sm"
              onClick={() => handleQuickPreset("corporate")}
              data-testid="preset-corporate"
            >
              Corporate
            </Button>
            <Button
              variant="outline"
              size="sm"
              onClick={() => handleQuickPreset("modern")}
              data-testid="preset-modern"
            >
              Modern
            </Button>
            <Button
              variant="outline"
              size="sm"
              onClick={() => handleQuickPreset("vibrant")}
              data-testid="preset-vibrant"
            >
              Vibrant
            </Button>
          </div>
        </div>

        <Separator />

        {/* Brand Identity */}
        <div className="space-y-4">
          <Label className="text-sm font-medium">Brand Identity</Label>
          
          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
            <div className="space-y-2">
              <Label htmlFor="brand-name" className="text-sm text-muted-foreground">
                Brand Name
              </Label>
              <Input
                id="brand-name"
                value={brand.name || ""}
                onChange={handleInputChange("name")}
                placeholder="Your brand name"
                data-testid="input-brand-name"
              />
            </div>

            <div className="space-y-2">
              <Label htmlFor="logo-url" className="text-sm text-muted-foreground flex items-center gap-1">
                <Image className="w-3 h-3" />
                Logo URL
              </Label>
              <Input
                id="logo-url"
                value={brand.logoUrl || ""}
                onChange={handleInputChange("logoUrl")}
                placeholder="https://example.com/logo.png"
                data-testid="input-logo-url"
              />
            </div>
          </div>
        </div>

        {/* Color Palette */}
        <div className="space-y-4">
          <Label className="text-sm font-medium">Color Palette</Label>
          
          <div className="grid grid-cols-3 gap-4">
            <div className="space-y-2">
              <Label htmlFor="primary-color" className="text-sm text-muted-foreground">
                Primary Color
              </Label>
              <div className="flex gap-2">
                <Input
                  id="primary-color"
                  type="color"
                  value={brand.primary}
                  onChange={handleInputChange("primary")}
                  className="w-12 h-10 p-1 cursor-pointer"
                  data-testid="input-primary-color"
                />
                <Input
                  value={brand.primary}
                  onChange={handleInputChange("primary")}
                  placeholder="#6D28D9"
                  className="flex-1"
                  data-testid="input-primary-hex"
                />
              </div>
            </div>

            <div className="space-y-2">
              <Label htmlFor="secondary-color" className="text-sm text-muted-foreground">
                Secondary Color
              </Label>
              <div className="flex gap-2">
                <Input
                  id="secondary-color"
                  type="color"
                  value={brand.secondary || "#06B6D4"}
                  onChange={handleInputChange("secondary")}
                  className="w-12 h-10 p-1 cursor-pointer"
                  data-testid="input-secondary-color"
                />
                <Input
                  value={brand.secondary || ""}
                  onChange={handleInputChange("secondary")}
                  placeholder="#06B6D4"
                  className="flex-1"
                  data-testid="input-secondary-hex"
                />
              </div>
            </div>

            <div className="space-y-2">
              <Label htmlFor="accent-color" className="text-sm text-muted-foreground">
                Accent Color
              </Label>
              <div className="flex gap-2">
                <Input
                  id="accent-color"
                  type="color"
                  value={brand.accent || "#F59E0B"}
                  onChange={handleInputChange("accent")}
                  className="w-12 h-10 p-1 cursor-pointer"
                  data-testid="input-accent-color"
                />
                <Input
                  value={brand.accent || ""}
                  onChange={handleInputChange("accent")}
                  placeholder="#F59E0B"
                  className="flex-1"
                  data-testid="input-accent-hex"
                />
              </div>
            </div>
          </div>

          {/* Palette Preview */}
          <div className="space-y-2">
            <Label className="text-sm text-muted-foreground">Generated Palette Preview</Label>
            <div className="flex gap-1 h-8" data-testid="palette-preview">
              {previewPalette.slice(0, 9).map((color, index) => (
                <div
                  key={index}
                  className="flex-1 rounded border"
                  style={{ backgroundColor: color }}
                  title={`Color ${index + 1}: ${color}`}
                />
              ))}
            </div>
            <p className="text-xs text-muted-foreground">
              These colors will be automatically applied to all charts and visualizations.
            </p>
          </div>
        </div>

        {/* Typography */}
        <div className="space-y-4">
          <Label className="text-sm font-medium flex items-center gap-2">
            <Type className="w-4 h-4" />
            Typography
          </Label>
          
          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
            <div className="space-y-2">
              <Label htmlFor="font-head" className="text-sm text-muted-foreground">
                Heading Font
              </Label>
              <Input
                id="font-head"
                value={brand.fontHead || ""}
                onChange={handleInputChange("fontHead")}
                placeholder="Inter, ui-sans-serif, system-ui"
                data-testid="input-font-head"
              />
            </div>

            <div className="space-y-2">
              <Label htmlFor="font-body" className="text-sm text-muted-foreground">
                Body Font
              </Label>
              <Input
                id="font-body"
                value={brand.fontBody || ""}
                onChange={handleInputChange("fontBody")}
                placeholder="Inter, ui-sans-serif, system-ui"
                data-testid="input-font-body"
              />
            </div>
          </div>

          {/* Font Preview */}
          <div className="space-y-2 p-3 bg-muted/30 rounded-lg">
            <div
              className="text-lg font-semibold"
              style={{ fontFamily: brand.fontHead, color: brand.primary }}
            >
              Heading Preview
            </div>
            <div
              className="text-sm"
              style={{ fontFamily: brand.fontBody }}
            >
              This is how your body text will appear in documents and charts.
            </div>
          </div>
        </div>

        {/* Brand Kit Status */}
        <div className="text-xs text-muted-foreground bg-muted/30 p-3 rounded-lg">
          <p className="font-medium mb-1">Brand Kit Status:</p>
          <ul className="space-y-1">
            <li>• Colors will be applied to all charts automatically</li>
            <li>• Cover designer will use your brand colors and fonts</li>
            <li>• Export functionality maintains brand consistency</li>
            <li>• Settings are saved in your browser</li>
          </ul>
        </div>
      </CardContent>
    </Card>
  );
}