import { useState, useEffect } from "react";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Badge } from "@/components/ui/badge";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { useBrandKitStore } from "@/stores/useBrandKitStore";
import { getBrandPalette, applyBrandKitStyles } from "@/lib/brand";
import { Palette, Plus, Trash2, Save } from "lucide-react";
import { useToast } from "@/hooks/use-toast";
import type { BrandKit } from "@shared/brand-types";

export function BrandKitManager() {
  const { kits, current, createKit, updateKit, deleteKit, setDefault, getCurrentKit, loadFromStorage } = useBrandKitStore();
  const { toast } = useToast();
  const [formData, setFormData] = useState<Omit<BrandKit, "id">>(current);
  const [isEditing, setIsEditing] = useState(false);

  useEffect(() => {
    loadFromStorage();
  }, [loadFromStorage]);

  useEffect(() => {
    const currentKit = getCurrentKit();
    setFormData(currentKit);
    applyBrandKitStyles(currentKit);
  }, [current, getCurrentKit]);

  const handleInputChange = (field: keyof Omit<BrandKit, "id">) => (
    e: React.ChangeEvent<HTMLInputElement>
  ) => {
    setFormData(prev => ({ ...prev, [field]: e.target.value }));
  };

  const handleSaveAsNew = () => {
    const id = createKit(formData);
    setIsEditing(false);
    toast({
      title: "Brand Kit Created",
      description: `New brand kit "${formData.name}" has been created and activated.`,
    });
  };

  const handleUpdate = () => {
    if (current.id) {
      updateKit(current.id, formData);
      setIsEditing(false);
      toast({
        title: "Brand Kit Updated",
        description: `Brand kit "${formData.name}" has been updated.`,
      });
    }
  };

  const handleDelete = (id: string) => {
    if (id === "default") {
      toast({
        title: "Cannot Delete",
        description: "The default brand kit cannot be deleted.",
        variant: "destructive",
      });
      return;
    }
    deleteKit(id);
    toast({
      title: "Brand Kit Deleted",
      description: "Brand kit has been deleted.",
    });
  };

  const handleSelectKit = (kitId: string) => {
    setDefault(kitId);
    const selectedKit = kits.find(k => k.id === kitId);
    if (selectedKit) {
      setFormData(selectedKit);
    }
  };

  const palette = getBrandPalette(formData);

  return (
    <Card className="w-full" data-testid="brand-kit-manager">
      <CardHeader>
        <CardTitle className="flex items-center gap-2">
          <Palette className="w-5 h-5 text-purple-500" />
          Brand Kit Manager
        </CardTitle>
      </CardHeader>

      <CardContent className="space-y-6">
        {/* Kit Selection */}
        <div className="space-y-2">
          <Label>Current Brand Kit</Label>
          <div className="flex gap-2">
            <Select value={current.id} onValueChange={handleSelectKit}>
              <SelectTrigger className="flex-1">
                <SelectValue />
              </SelectTrigger>
              <SelectContent>
                {kits.map((kit) => (
                  <SelectItem key={kit.id} value={kit.id}>
                    {kit.name} {kit.id === current.id && <Badge variant="outline" className="ml-2">Active</Badge>}
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>
            {current.id !== "default" && (
              <Button
                variant="destructive"
                size="sm"
                onClick={() => handleDelete(current.id)}
                data-testid="delete-kit-button"
              >
                <Trash2 className="w-4 h-4" />
              </Button>
            )}
          </div>
        </div>

        {/* Brand Identity */}
        <div className="space-y-4">
          <Label className="text-sm font-medium">Brand Identity</Label>
          
          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
            <div className="space-y-2">
              <Label htmlFor="brand-name" className="text-sm text-muted-foreground">
                Brand Name
              </Label>
              <Input
                id="brand-name"
                value={formData.name || ""}
                onChange={handleInputChange("name")}
                placeholder="Your brand name"
                data-testid="input-brand-name"
              />
            </div>

            <div className="space-y-2">
              <Label htmlFor="logo-url" className="text-sm text-muted-foreground">
                Logo URL
              </Label>
              <Input
                id="logo-url"
                value={formData.logoUrl || ""}
                onChange={handleInputChange("logoUrl")}
                placeholder="https://example.com/logo.png"
                data-testid="input-logo-url"
              />
            </div>
          </div>
        </div>

        {/* Color Palette */}
        <div className="space-y-4">
          <Label className="text-sm font-medium">Color Palette</Label>
          
          <div className="grid grid-cols-3 gap-4">
            <div className="space-y-2">
              <Label htmlFor="primary-color" className="text-sm text-muted-foreground">
                Primary
              </Label>
              <div className="flex gap-2">
                <Input
                  type="color"
                  value={formData.primary}
                  onChange={handleInputChange("primary")}
                  className="w-12 h-10 p-1"
                  data-testid="input-primary-color"
                />
                <Input
                  value={formData.primary}
                  onChange={handleInputChange("primary")}
                  className="flex-1"
                  data-testid="input-primary-hex"
                />
              </div>
            </div>

            <div className="space-y-2">
              <Label htmlFor="secondary-color" className="text-sm text-muted-foreground">
                Secondary
              </Label>
              <div className="flex gap-2">
                <Input
                  type="color"
                  value={formData.secondary || "#06B6D4"}
                  onChange={handleInputChange("secondary")}
                  className="w-12 h-10 p-1"
                  data-testid="input-secondary-color"
                />
                <Input
                  value={formData.secondary || ""}
                  onChange={handleInputChange("secondary")}
                  placeholder="#06B6D4"
                  className="flex-1"
                  data-testid="input-secondary-hex"
                />
              </div>
            </div>

            <div className="space-y-2">
              <Label htmlFor="accent-color" className="text-sm text-muted-foreground">
                Accent
              </Label>
              <div className="flex gap-2">
                <Input
                  type="color"
                  value={formData.accent || "#F59E0B"}
                  onChange={handleInputChange("accent")}
                  className="w-12 h-10 p-1"
                  data-testid="input-accent-color"
                />
                <Input
                  value={formData.accent || ""}
                  onChange={handleInputChange("accent")}
                  placeholder="#F59E0B"
                  className="flex-1"
                  data-testid="input-accent-hex"
                />
              </div>
            </div>
          </div>

          {/* Palette Preview */}
          <div className="space-y-2">
            <Label className="text-sm text-muted-foreground">Generated Palette</Label>
            <div className="flex gap-1 h-8" data-testid="palette-preview">
              {palette.slice(0, 6).map((color, index) => (
                <div
                  key={index}
                  className="flex-1 rounded border"
                  style={{ backgroundColor: color }}
                  title={`Color ${index + 1}: ${color}`}
                />
              ))}
            </div>
          </div>
        </div>

        {/* Typography */}
        <div className="space-y-4">
          <Label className="text-sm font-medium">Typography</Label>
          
          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
            <div className="space-y-2">
              <Label htmlFor="font-head" className="text-sm text-muted-foreground">
                Heading Font
              </Label>
              <Input
                id="font-head"
                value={formData.fontHead || ""}
                onChange={handleInputChange("fontHead")}
                placeholder="Inter, ui-sans-serif"
                data-testid="input-font-head"
              />
            </div>

            <div className="space-y-2">
              <Label htmlFor="font-body" className="text-sm text-muted-foreground">
                Body Font
              </Label>
              <Input
                id="font-body"
                value={formData.fontBody || ""}
                onChange={handleInputChange("fontBody")}
                placeholder="Inter, ui-sans-serif"
                data-testid="input-font-body"
              />
            </div>
          </div>
        </div>

        {/* Actions */}
        <div className="flex justify-between items-center pt-4 border-t">
          <div className="flex gap-2">
            <Button
              onClick={handleSaveAsNew}
              className="flex items-center gap-2"
              data-testid="save-as-new-button"
            >
              <Plus className="w-4 h-4" />
              Save as New
            </Button>
            
            <Button
              onClick={handleUpdate}
              variant="outline"
              className="flex items-center gap-2"
              data-testid="update-button"
            >
              <Save className="w-4 h-4" />
              Update Current
            </Button>
          </div>
          
          <div className="text-xs text-muted-foreground">
            {kits.length} brand kit{kits.length !== 1 ? 's' : ''} • Auto-applied to charts & covers
          </div>
        </div>
      </CardContent>
    </Card>
  );
}