import React, { useState } from 'react';
import { useLocation } from 'wouter';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogHeader,
  DialogTitle,
} from '@/components/ui/dialog';
import { useAuth } from '@/contexts/AuthContext';
import { useToast } from '@/hooks/use-toast';
import { handleAuthSuccess, getUserPaidStatus } from '@/services/authRedirect';
import { ArrowLeft } from 'lucide-react';
import { useLogo } from '@/hooks/useLogo';
import { sendPasswordResetEmail } from 'firebase/auth';
import { auth } from '@/lib/firebase';

interface AuthModalProps {
  isOpen: boolean;
  onClose: () => void;
}

export function AuthModal({ isOpen, onClose }: AuthModalProps) {
  const [isLogin, setIsLogin] = useState(true);
  const [isForgotPassword, setIsForgotPassword] = useState(false);
  const [email, setEmail] = useState('');
  const [resetEmail, setResetEmail] = useState('');
  const [password, setPassword] = useState('');
  const [displayName, setDisplayName] = useState('');
  const [loading, setLoading] = useState(false);
  const logoPath = useLogo();
  
  const { login, signup } = useAuth();
  const { toast } = useToast();
  const [, navigate] = useLocation();

  async function handleSubmit(e: React.FormEvent) {
    e.preventDefault();
    
    if (!email || !password || (!isLogin && !displayName)) {
      toast({
        title: "Error",
        description: "Please fill in all fields",
        variant: "destructive"
      });
      return;
    }

    try {
      setLoading(true);
      if (isLogin) {
        await login(email, password);
      } else {
        await signup(email, password, displayName);
      }
      
      // Success - handle post-authentication redirect
      const isPaid = getUserPaidStatus();
      handleAuthSuccess({ 
        isPaid, 
        newUser: !isLogin,  // true if this was a signup, false if login
        email: email,
        navigate 
      });
      
      // Reset form state
      setEmail('');
      setPassword('');
      setDisplayName('');
      setIsLogin(true);
      
      // Close modal
      onClose();
    } catch (error: any) {
      // Transform Firebase errors into user-friendly messages
      let errorMessage = "An error occurred. Please try again.";
      
      if (error.code) {
        switch (error.code) {
          case 'auth/invalid-credential':
          case 'auth/wrong-password':
          case 'auth/user-not-found':
            errorMessage = "Invalid email or password. Please check your credentials and try again.";
            break;
          case 'auth/email-already-in-use':
            errorMessage = "This email is already registered. Please sign in instead.";
            break;
          case 'auth/weak-password':
            errorMessage = "Password is too weak. Please use at least 6 characters.";
            break;
          case 'auth/invalid-email':
            errorMessage = "Invalid email address. Please check and try again.";
            break;
          case 'auth/user-disabled':
            errorMessage = "This account has been disabled. Please contact support.";
            break;
          case 'auth/too-many-requests':
            errorMessage = "Too many failed attempts. Please try again later.";
            break;
          default:
            errorMessage = error.message || "An error occurred. Please try again.";
        }
      } else {
        errorMessage = error.message || "An error occurred. Please try again.";
      }
      
      toast({
        title: "Authentication Error",
        description: errorMessage,
        variant: "destructive"
      });
    } finally {
      setLoading(false);
    }
  }

  async function handleForgotPassword(e: React.FormEvent) {
    e.preventDefault();
    
    if (!resetEmail) {
      toast({
        title: "Error",
        description: "Please enter your email address",
        variant: "destructive"
      });
      return;
    }

    try {
      setLoading(true);
      await sendPasswordResetEmail(auth, resetEmail);
      
      // Generic success message to prevent account enumeration
      toast({
        title: "Reset Link Sent",
        description: "If an account with this email exists, a reset link has been sent.",
        variant: "default"
      });
      
      // Reset form and go back to login
      setResetEmail('');
      setIsForgotPassword(false);
      
    } catch (error: any) {
      // Generic error message to prevent account enumeration
      toast({
        title: "Reset Link Sent",
        description: "If an account with this email exists, a reset link has been sent.",
        variant: "default"
      });
    } finally {
      setLoading(false);
    }
  }

  function handleForgotPasswordClick(e: React.MouseEvent) {
    e.preventDefault();
    // Prefill reset email with current email input for better UX
    setResetEmail(email);
    setIsForgotPassword(true);
  }

  function handleBackToLogin() {
    setIsForgotPassword(false);
    setResetEmail('');
  }

  // Clean up modal state when closing
  function handleClose() {
    setIsForgotPassword(false);
    setResetEmail('');
    onClose();
  }

  return (
    <Dialog open={isOpen} onOpenChange={(open) => { if (!open) handleClose(); }}>
      <DialogContent className="sm:max-w-md">
        <DialogHeader className="text-center">
          <div className="flex justify-center mb-4">
            <img src={logoPath} alt="IBrandBiz Logo" className="h-12 w-auto" data-testid="img-modal-logo" />
          </div>
          <DialogTitle className="text-xl font-semibold">
            {isForgotPassword ? 'Reset Password' : 'Welcome to IBrandBiz'}
          </DialogTitle>
          <DialogDescription>
            {isForgotPassword ? 'Enter your email to receive a reset link' : 'Your complete business branding toolkit'}
          </DialogDescription>
        </DialogHeader>

        <div className="mt-6">
          {/* Back button for forgot password */}
          {isForgotPassword && (
            <div className="mb-4">
              <button
                type="button"
                onClick={handleBackToLogin}
                className="flex items-center text-sm text-primary hover:underline"
                data-testid="button-back-to-login"
              >
                <ArrowLeft className="h-4 w-4 mr-1" />
                Back to login
              </button>
            </div>
          )}

          {/* Auth Toggle - only show when not in forgot password mode */}
          {!isForgotPassword && (
            <div className="flex mb-6 bg-muted rounded-lg p-1">
              <button
                type="button"
                className={`flex-1 py-2 px-4 rounded-md text-sm font-medium transition-all ${
                  isLogin 
                    ? 'bg-primary hover:bg-primary/90 text-white font-bold' 
                    : 'text-muted-foreground hover:text-foreground'
                }`}
                onClick={() => setIsLogin(true)}
                data-testid="button-modal-login-tab"
              >
                Sign In
              </button>
              <button
                type="button"
                className={`flex-1 py-2 px-4 rounded-md text-sm font-medium transition-all ${
                  !isLogin 
                    ? 'bg-primary hover:bg-primary/90 text-white font-bold' 
                    : 'text-muted-foreground hover:text-foreground'
                }`}
                onClick={() => setIsLogin(false)}
                data-testid="button-modal-signup-tab"
              >
                Sign Up
              </button>
            </div>
          )}

          {/* Forgot Password Form */}
          {isForgotPassword ? (
            <form onSubmit={handleForgotPassword} className="space-y-4">
              <div>
                <Label htmlFor="modal-reset-email" className="block text-sm font-medium text-foreground mb-2">
                  Email Address
                </Label>
                <Input
                  id="modal-reset-email"
                  type="email"
                  value={resetEmail}
                  onChange={(e) => setResetEmail(e.target.value)}
                  placeholder="Enter your email address"
                  data-testid="input-modal-reset-email"
                />
              </div>
              <Button 
                type="submit" 
                className="w-full bg-primary hover:bg-primary/90 text-white font-bold"
                disabled={loading}
                data-testid="button-send-reset-link"
              >
                {loading ? 'Sending...' : 'Send Reset Link'}
              </Button>
            </form>
          ) : (
            /* Login/Signup Form */
            <form onSubmit={handleSubmit} className="space-y-4">
              {!isLogin && (
                <div>
                  <Label htmlFor="modal-displayName" className="block text-sm font-medium text-foreground mb-2">
                    Full Name
                  </Label>
                  <Input
                    id="modal-displayName"
                    type="text"
                    value={displayName}
                    onChange={(e) => setDisplayName(e.target.value)}
                    placeholder="Enter your full name"
                    data-testid="input-modal-displayname"
                  />
                </div>
              )}
              <div>
                <Label htmlFor="modal-email" className="block text-sm font-medium text-foreground mb-2">
                  Email
                </Label>
                <Input
                  id="modal-email"
                  type="email"
                  value={email}
                  onChange={(e) => setEmail(e.target.value)}
                  placeholder="Enter your email"
                  data-testid="input-modal-email"
                />
              </div>
              <div>
                <Label htmlFor="modal-password" className="block text-sm font-medium text-foreground mb-2">
                  Password
                </Label>
                <Input
                  id="modal-password"
                  type="password"
                  value={password}
                  onChange={(e) => setPassword(e.target.value)}
                  placeholder={isLogin ? "Enter your password" : "Create a password"}
                  data-testid="input-modal-password"
                />
              </div>
              <Button 
                type="submit" 
                className="w-full bg-primary hover:bg-primary/90 text-white font-bold"
                disabled={loading}
                data-testid="button-modal-submit"
              >
                {loading ? 'Please wait...' : (isLogin ? 'Sign In' : 'Create Account')}
              </Button>
            </form>
          )}

          {/* Forgot password link - only show in login mode */}
          {!isForgotPassword && isLogin && (
            <div className="mt-4 text-center">
              <button
                type="button"
                onClick={handleForgotPasswordClick}
                className="text-sm text-primary hover:underline"
                data-testid="link-forgot-password"
              >
                Forgot your password?
              </button>
            </div>
          )}
        </div>
      </DialogContent>
    </Dialog>
  );
}