import { useAuth } from '@/contexts/AuthContext';
import { User } from 'firebase/auth';

/**
 * Checks if an email is in the list of configured owner emails
 */
export function isOwnerEmail(email: string): boolean {
  if (!email) return false;
  
  // Get owner emails from environment variable
  const ownerEmails = import.meta.env.VITE_IBRANDBIZ_OWNER_EMAILS || '';
  if (!ownerEmails) {
    // Development fallback
    if (import.meta.env.MODE === 'development') {
      console.log('⚠️ VITE_IBRANDBIZ_OWNER_EMAILS not configured. Using default owner email.');
      return email.toLowerCase() === 'jrichards@ibrandbiz.com';
    }
    return false;
  }
  
  // Parse comma-separated list and normalize emails
  const ownerList = ownerEmails
    .split(',')
    .map(e => e.trim().toLowerCase())
    .filter(e => e.length > 0);
  
  return ownerList.includes(email.toLowerCase());
}

/**
 * Get admin authentication headers for API requests
 * Returns appropriate headers based on user type (owner vs admin)
 */
export async function getAdminAuthHeaders(currentUser?: User | null): Promise<Record<string, string>> {
  // Check if current user is owner
  if (currentUser?.email && isOwnerEmail(currentUser.email)) {
    try {
      const token = await currentUser.getIdToken();
      console.log('🔐 Using Firebase token for owner:', currentUser.email);
      return {
        'Authorization': `Bearer ${token}`
      };
    } catch (error) {
      console.error('Failed to get Firebase token for owner:', error);
    }
  }
  
  // Fall back to admin key for regular admins
  const adminKey = localStorage.getItem("adminKey") || "";
  if (!adminKey) {
    throw new Error("Admin authentication required. Owner users should be authenticated via Firebase, others need admin key.");
  }
  
  console.log('🔐 Using admin key for non-owner admin');
  return {
    'x-admin-key': adminKey
  };
}

/**
 * Hook to get admin authentication status and method
 */
export function useAdminAuth() {
  const { currentUser } = useAuth();
  
  const isOwner = currentUser?.email ? isOwnerEmail(currentUser.email) : false;
  const hasAdminKey = !!localStorage.getItem("adminKey");
  
  const isAuthenticated = isOwner || hasAdminKey;
  const authMethod = isOwner ? 'owner' : hasAdminKey ? 'key' : 'none';
  
  // Create a bound version of getAdminAuthHeaders
  const getAuthHeaders = () => getAdminAuthHeaders(currentUser);
  
  return {
    isAuthenticated,
    isOwner,
    hasAdminKey,
    authMethod,
    getAuthHeaders
  };
}