import React, { Suspense } from "react";
import { Route, Switch, useLocation, useParams } from "wouter";
import { queryClient } from "./lib/queryClient";
import { QueryClientProvider } from "@tanstack/react-query";
import { Toaster } from "@/components/ui/toaster";
import { Toaster as SonnerToaster } from "sonner";
import { TooltipProvider } from "@/components/ui/tooltip";
import { AuthProvider } from "@/contexts/AuthContext";
import { CartProvider } from "@/contexts/CartContext";
import { NotificationsProvider } from "@/components/NotificationsProvider";
import { ProtectedRoute } from "@/components/ProtectedRoute";
import { Sidebar } from "@/components/Sidebar";
import { TopBar } from "@/components/TopBar";
import { DashboardTemplatePage } from "@/components/DashboardTemplatePage";
import Footer from "@/components/Footer";
import { CookieConsentBanner } from "@/components/CookieConsentBanner";
import { featureFlags } from "@/lib/featureFlags";
import { useIsMobile } from "@/hooks/useIsMobile";
import { useAuth } from "@/contexts/AuthContext";
import Home from "@/pages/Home";
import About from "@/pages/About";
import ServicesPage from "@/pages/Services";
import Products from "@/pages/Products";
import Terms from "@/pages/Terms";
import Privacy from "@/pages/Privacy";
import Cookies from "@/pages/Cookies";
import Dashboard from "@/pages/Dashboard";
import MobileMarketingLayout from "@/layouts/MobileMarketingLayout";
import BrandKit from "@/pages/BrandKit";
import BusinessDevelopmentPricing from "@/pages/pricing/BusinessDevelopmentPricing";
import WebServicesPricing from "@/pages/pricing/WebServicesPricing";
import WebAddons from "@/pages/pricing/WebAddons";
import BusinessName from "@/pages/BusinessName";
import SloganPage from "@/pages/SloganPage";
import BusinessPlanBuilder from "@/pages/BusinessPlanBuilder";
import BuilderV7 from "@/pages/Builder.v7";
import DomainsPage from "@/pages/DomainsPage";
import DomainOrderStatusPage from "@/pages/DomainOrderStatusPage";
import SocialMedia from "@/pages/SocialMedia";
import WebsiteTemplates from "@/pages/WebsiteTemplates";
import Hosting from "@/pages/Hosting";
import WebHosting from "@/pages/WebHosting";
import StockImages from "@/pages/StockImages";
import BrandAssets from "@/pages/BrandAssets";
import ResourcesPage from "@/pages/Resources/ResourcesPage";
import FAQPage from "@/pages/FAQPage";
import SettingsPage from "@/pages/Settings/SettingsPage";
import MyPurchases from "@/pages/Account/MyPurchases";
import SignupPage from "@/pages/SignupPage";
import Redirect from "@/components/Redirect";
import { tlog } from "@/utils/telemetry";
import AdminHome from "@/pages/Admin/AdminHome";
import AdminLogin from "@/pages/Admin/AdminLogin";
import AdminDashboard from "@/pages/Admin/AdminDashboard";
import AdminAssetsReview from "@/pages/Admin/AdminAssetsReview";
import AdminMarketplace from "@/pages/Admin/AdminMarketplace";
import AdminCoverTemplates from "@/pages/Admin/AdminCoverTemplates";
import AdminInfographicTemplates from "@/pages/Admin/AdminInfographicTemplates";
import AdminValidateManifest from "@/pages/Admin/AdminValidateManifest";
import AdminUploadLogoTemplate from "@/pages/Admin/AdminUploadLogoTemplate";
import AdminUploadBusinessPlanFirebase from "@/pages/Admin/AdminUploadBusinessPlanFirebase";
import AdminValidateBPManifest from "@/pages/Admin/AdminValidateBPManifest";
import AdminImportBusinessPlans from "@/pages/Admin/AdminImportBusinessPlans";

// Lazy
const IconLabeler = React.lazy(() => import("@/pages/Admin/IconLabeler"));
const StockPhotos = React.lazy(() => import("@/pages/business-assets/stock/StockPhotos"));
const Mockups = React.lazy(() => import("@/pages/business-assets/stock/Mockups"));

import ProfilePage from "@/pages/Profile/ProfilePage";
import NotFound from "@/pages/not-found";

// Creator Marketplace pages
import CreatorDashboard from "@/pages/Creator/CreatorDashboard";
import CreatorOnboarding from "@/pages/Creator/CreatorOnboarding";
import CreatorUpload from "@/pages/Creator/CreatorUpload";
import CreatorAssets from "@/pages/Creator/CreatorAssets";
import CreatorEarnings from "@/pages/Creator/CreatorEarnings";
import MyMarketplacePage from "@/pages/Creator/MyMarketplacePage"; // NEW
import UploadCoverTemplate from "@/pages/Creator/UploadCoverTemplate";
import UploadInfographicTemplate from "@/pages/Creator/UploadInfographicTemplate";

// Public marketplace
import MarketplacePage from "@/pages/MarketplacePage";
import AssetDetailPage from "@/pages/AssetDetailPage";
import CreatorProfilePage from "@/pages/CreatorProfilePage";

// Business Assets
import PresentationTemplate from "@/pages/business-assets/templates/PresentationTemplate";
import BusinessPlanTemplatesPage from "@/pages/Public/BusinessPlanTemplatesPage";
import BusinessPlanTemplateDetailsPage from "@/pages/Public/BusinessPlanTemplateDetailsPage";
import LogoTemplatesPage from "@/pages/business-assets/templates/LogoTemplates";
import AILogoCreator from "@/pages/business-development/AILogoCreator";
import LogoCustomizerPage from "@/pages/LogoCustomizerPage";
import LogoTemplateDetail from "@/pages/logo-templates/LogoTemplateDetail";
import IconsPage from "@/pages/business-assets/stock/Icons";

// Dev Tools
import LinkAuditPage from "@/pages/dev/LinkAuditPage";
import DiagnosticsPage from "@/pages/dev/DiagnosticsPage";

function CookieRedirect() {
  window.location.href = "/cookie-policy";
  return null;
}

function PlanRedirect() {
  window.location.href = "/business-development/ai-business-plan-developer";
  return null;
}

function LogoTemplateRedirect() {
  const { oldId } = useParams<{ oldId: string }>();
  const composerId = oldId?.startsWith('logo-wordmark-') 
    ? `logo-composer-${oldId.replace('logo-wordmark-', '')}`
    : `logo-composer-${oldId}`;
  window.location.href = `/brand-development/ai-logo-creator/${composerId}`;
  return null;
}


function AppLayout({ children, title }: { children: React.ReactNode; title: string }) {
  return (
    <div className="min-h-screen bg-background flex flex-col">
      <div className="flex flex-1 min-h-0 items-stretch">
        <Sidebar />
        <div className="flex-1 min-w-0">
          <TopBar title={title} />
          <main className="p-6">{children}</main>
        </div>
      </div>
      <Footer />
    </div>
  );
}

function Router() {
  const isMobile = useIsMobile(); // < 1024px
  const { currentUser } = useAuth();

  // For mobile non-logged-in users, show mobile marketing layout for marketing routes
  const isMobileMarketing = isMobile && !currentUser;
  const marketingRoutes = ['/', '/services', '/products', '/pricing', '/faq', '/about', '/terms', '/privacy', '/resources'];

  const [location] = useLocation();
  const isMarketingRoute = marketingRoutes.some(route => 
    route === '/' ? location === route : location.startsWith(route)
  );

  // Mobile marketing layout for non-logged-in users on marketing routes
  if (isMobileMarketing && isMarketingRoute) {
    return <MobileMarketingLayout />;
  }

  return (
    <Switch>
      {/* Public */}
      <Route path="/"><Home /></Route>
      <Route path="/about"><About /></Route>
      <Route path="/services"><ServicesPage /></Route>
      <Route path="/products"><Products /></Route>
      <Route path="/terms"><Terms /></Route>
      <Route path="/privacy"><Privacy /></Route>
      <Route path="/cookie-policy"><Cookies /></Route>
      <Route path="/cookies"><Cookies /></Route>
      {/* Pricing Routes */}
      <Route path="/pricing/business-development"><BusinessDevelopmentPricing /></Route>
      <Route path="/pricing/web-services"><WebServicesPricing /></Route>
      <Route path="/pricing/addons"><WebAddons /></Route>
      
      {/* Legacy Pricing Redirects */}
      <Route path="/pricing/assets"><Redirect to="/pricing/business-development" from="/pricing/assets" /></Route>
      <Route path="/pricing"><Redirect to="/pricing/business-development" from="/pricing" /></Route>
      
      {/* Legacy Path Redirects */}
      <Route path="/domains"><Redirect to="/web-services/domains" from="/domains" /></Route>
      <Route path="/website-builder"><Redirect to="/website-templates" from="/website-builder" /></Route>
      <Route path="/brand-development/brand-voice-generator"><Redirect to="/brand-kits" from="/brand-development/brand-voice-generator" /></Route>
      <Route path="/brand-development/brand-kit-generator"><Redirect to="/brand-kits" from="/brand-development/brand-kit-generator" /></Route>
      <Route path="/social-content/infographics-library"><Redirect to="/social-kits" from="/social-content/infographics-library" /></Route>
      
      <Route path="/signup"><SignupPage /></Route>
      <Route path="/resources"><ResourcesPage /></Route>
      <Route path="/faq"><FAQPage /></Route>
      <Route path="/web-services/web-hosting"><WebHosting /></Route>
      <Route path="/stock-images">
        <ProtectedRoute>
          <Suspense fallback={<div className="flex items-center justify-center h-64"><div className="animate-spin h-8 w-8 border-2 border-primary border-t-transparent rounded-full"></div></div>}>
            <StockPhotos />
          </Suspense>
        </ProtectedRoute>
      </Route>

      {/* Public Marketplace */}
      <Route path="/marketplace">
        <AppLayout title="Creator Marketplace">
          <MarketplacePage />
        </AppLayout>
      </Route>
      <Route path="/marketplace/asset/:id">
        <AppLayout title="Asset Details">
          <AssetDetailPage />
        </AppLayout>
      </Route>
      <Route path="/marketplace/creator/:id">
        <AppLayout title="Creator Profile">
          <CreatorProfilePage />
        </AppLayout>
      </Route>

      {/* Admin */}
      <Route path="/admin"><AdminHome /></Route>
      <Route path="/admin/login"><AdminLogin /></Route>
      <Route path="/admin/dashboard"><AdminDashboard /></Route>
      <Route path="/admin/assets"><AdminAssetsReview /></Route>
      <Route path="/admin/marketplace">
        <DashboardTemplatePage title="Marketplace Management">
          <AdminMarketplace />
        </DashboardTemplatePage>
      </Route>
      <Route path="/admin/cover-templates">
        <DashboardTemplatePage title="Cover Templates Management">
          <AdminCoverTemplates />
        </DashboardTemplatePage>
      </Route>
      <Route path="/admin/infographic-templates">
        <DashboardTemplatePage title="Infographic Templates Management">
          <AdminInfographicTemplates />
        </DashboardTemplatePage>
      </Route>
      <Route path="/admin/validate-logo-manifest"><AdminValidateManifest /></Route>
      <Route path="/admin/validate-bp-manifest"><AdminValidateBPManifest /></Route>
      <Route path="/admin/upload-logo-template"><AdminUploadLogoTemplate /></Route>
      <Route path="/admin/upload-business-plan-firebase"><AdminUploadBusinessPlanFirebase /></Route>
      <Route path="/admin/import-bp-templates"><AdminImportBusinessPlans /></Route>

      {/* Settings */}
      <Route path="/settings">
        <ProtectedRoute>
          <DashboardTemplatePage title="Settings">
            <SettingsPage />
          </DashboardTemplatePage>
        </ProtectedRoute>
      </Route>
      
      <Route path="/account/purchases">
        <ProtectedRoute>
          <MyPurchases />
        </ProtectedRoute>
      </Route>

      {/* Dashboard/Profile */}
      <Route path="/dashboard">
        <ProtectedRoute>
          <AppLayout title="Dashboard"><Dashboard /></AppLayout>
        </ProtectedRoute>
      </Route>
      <Route path="/profile">
        <ProtectedRoute>
          <AppLayout title="Profile"><ProfilePage /></AppLayout>
        </ProtectedRoute>
      </Route>

      {/* App Core */}
      <Route path="/business-development/brand-name-wizard">
        <ProtectedRoute>
          <AppLayout title="Brand Name Wizard"><BusinessName /></AppLayout>
        </ProtectedRoute>
      </Route>
      <Route path="/business-development/ai-logo-creator">
        <ProtectedRoute>
          <AILogoCreator />
        </ProtectedRoute>
      </Route>
      <Route path="/business-development/brand-kit">
        <ProtectedRoute>
          <AppLayout title="Brand Kit"><BrandKit /></AppLayout>
        </ProtectedRoute>
      </Route>
      <Route path="/brand-kits">
        <ProtectedRoute>
          <AppLayout title="Brand Kit"><BrandKit /></AppLayout>
        </ProtectedRoute>
      </Route>
      <Route path="/web-services/domains">
        <ProtectedRoute>
          <AppLayout title="Domains"><DomainsPage /></AppLayout>
        </ProtectedRoute>
      </Route>
      <Route path="/web-services/domains/orders/:orderId">
        <ProtectedRoute>
          <AppLayout title="Domain Order Status"><DomainOrderStatusPage /></AppLayout>
        </ProtectedRoute>
      </Route>
      <Route path="/website-templates">
        <ProtectedRoute>
          <AppLayout title="Website Templates"><WebsiteTemplates /></AppLayout>
        </ProtectedRoute>
      </Route>
      <Route path="/social-kits">
        <ProtectedRoute>
          <AppLayout title="Social Media Kits"><SocialMedia /></AppLayout>
        </ProtectedRoute>
      </Route>
      <Route path="/social-media-kits">
        <ProtectedRoute>
          <AppLayout title="Social Media Kits"><SocialMedia /></AppLayout>
        </ProtectedRoute>
      </Route>
      <Route path="/hosting">
        <ProtectedRoute>
          <AppLayout title="Web Hosting"><Hosting /></AppLayout>
        </ProtectedRoute>
      </Route>

      {/* Feature Flags */}
      {featureFlags.sloganGenerator && (
        <Route path="/business-development/slogan-generator">
          <ProtectedRoute>
            <AppLayout title="Slogan Generator"><SloganPage /></AppLayout>
          </ProtectedRoute>
        </Route>
      )}
      {/* Redirect old /plan route to new location */}
      <Route path="/plan">
        <PlanRedirect />
      </Route>

      {featureFlags.businessPlan && (
        <Route path="/business-development/ai-business-plan-developer">
          <ProtectedRoute>
            <BusinessPlanBuilder />
          </ProtectedRoute>
        </Route>
      )}

      {/* Builder V7 */}
      <Route path="/builder">
        <ProtectedRoute>
          <AppLayout title="AI Frameworks Builder V7"><BuilderV7 /></AppLayout>
        </ProtectedRoute>
      </Route>

      {/* Business Assets */}
      <Route path="/business-assets/templates/presentation-template">
        <ProtectedRoute><PresentationTemplate /></ProtectedRoute>
      </Route>
      <Route path="/business-assets/templates/business-plan"><BusinessPlanTemplatesPage /></Route>
      <Route path="/business-assets/templates/business-plan/:slug"><BusinessPlanTemplateDetailsPage /></Route>
      <Route path="/business-assets/logo-templates">
        <ProtectedRoute><LogoTemplatesPage /></ProtectedRoute>
      </Route>
      <Route path="/business-assets/templates/logo">
        <ProtectedRoute><LogoTemplatesPage /></ProtectedRoute>
      </Route>
      <Route path="/brand-development/ai-logo-creator/:composerId">
        <ProtectedRoute><LogoCustomizerPage /></ProtectedRoute>
      </Route>
      {/* Redirect old logo template path to new location */}
      <Route path="/logo-templates/:oldId">
        <LogoTemplateRedirect />
      </Route>

      {/* Creator Program */}
      <Route path="/creator/dashboard">
        <ProtectedRoute>
          <DashboardTemplatePage title="Creator Dashboard"><CreatorDashboard /></DashboardTemplatePage>
        </ProtectedRoute>
      </Route>
      <Route path="/creator/onboarding">
        <ProtectedRoute>
          <DashboardTemplatePage title="Creator Onboarding"><CreatorOnboarding /></DashboardTemplatePage>
        </ProtectedRoute>
      </Route>
      <Route path="/creator/upload">
        <ProtectedRoute>
          <DashboardTemplatePage title="Upload Asset"><CreatorUpload /></DashboardTemplatePage>
        </ProtectedRoute>
      </Route>
      <Route path="/creator/upload-cover-template">
        <ProtectedRoute>
          <UploadCoverTemplate />
        </ProtectedRoute>
      </Route>
      <Route path="/creator/upload-infographic-template">
        <ProtectedRoute>
          <UploadInfographicTemplate />
        </ProtectedRoute>
      </Route>
      <Route path="/creator/assets">
        <ProtectedRoute>
          <DashboardTemplatePage title="Manage Assets"><CreatorAssets /></DashboardTemplatePage>
        </ProtectedRoute>
      </Route>
      <Route path="/creator/earnings">
        <ProtectedRoute>
          <DashboardTemplatePage title="Creator Earnings"><CreatorEarnings /></DashboardTemplatePage>
        </ProtectedRoute>
      </Route>

      {/* NEW: My Marketplace (creator self view) */}
      <Route path="/creator/my-marketplace">
        <ProtectedRoute>
          <DashboardTemplatePage title="My Marketplace">
            <MyMarketplacePage />
          </DashboardTemplatePage>
        </ProtectedRoute>
      </Route>

      {/* Stock Library (lazy) */}
      <Route path="/business-assets/stock/icons">
        <ProtectedRoute><IconsPage /></ProtectedRoute>
      </Route>
      <Route path="/business-assets/stock/mockups">
        <ProtectedRoute>
          <Suspense fallback={<div className="flex items-center justify-center h-64"><div className="animate-spin h-8 w-8 border-2 border-primary border-t-transparent rounded-full"></div></div>}>
            <Mockups />
          </Suspense>
        </ProtectedRoute>
      </Route>
      <Route path="/business-assets/stock/photos">
        <ProtectedRoute>
          <Suspense fallback={<div className="flex items-center justify-center h-64"><div className="animate-spin h-8 w-8 border-2 border-primary border-t-transparent rounded-full"></div></div>}>
            <StockPhotos />
          </Suspense>
        </ProtectedRoute>
      </Route>

      {/* Template Libraries */}

      {/* Catch-all for unknown pricing routes */}
      <Route path="/pricing/:unknown*"><Redirect to="/pricing/business-development" from="/pricing/*" /></Route>

      {/* Dev Tools (only available in development) */}
      <Route path="/__qa/diag"><DiagnosticsPage /></Route>
      <Route path="/__qa/link-audit"><LinkAuditPage /></Route>

      {/* 404 */}
      <Route path="*"><NotFound /></Route>
    </Switch>
  );
}

function App() {
  return (
    <QueryClientProvider client={queryClient}>
      <TooltipProvider>
        <AuthProvider>
          <CartProvider>
            <NotificationsProvider>
              <Toaster />
              <SonnerToaster position="top-right" />
              <Router />
              <CookieConsentBanner />
            </NotificationsProvider>
          </CartProvider>
        </AuthProvider>
      </TooltipProvider>
    </QueryClientProvider>
  );
}

export default App;