/**
 * mcp-server.js
 * Minimal MCP-style bridge for IBrandBiz (dev/tools) on Replit.
 *
 * - Auth: Bearer token (MCP_BEARER_TOKEN)
 * - Whitelist: commands that may be executed
 * - Audit log: ./mcp-audit.log
 * - GitHub PRs: requires GITHUB_TOKEN env for repo operations
 *
 * NOTE: This is intended for dev usage. Do NOT expose to public without firewalling.
 */

const express = require("express");
const bodyParser = require("body-parser");
const { exec } = require("child_process");
const fs = require("fs").promises;
const path = require("path");
const fetch = require("node-fetch"); // lightweight fetch for GitHub calls
const app = express();

const PORT = process.env.PORT || 3000;
const BEARER = process.env.MCP_BEARER_TOKEN || "changeme";
const GITHUB_TOKEN = process.env.GITHUB_TOKEN || null;
const REPO_OWNER = process.env.REPO_OWNER || null; // e.g. "your-org-or-user"
const REPO_NAME = process.env.REPO_NAME || null;   // e.g. "ibrandbiz"

app.use(bodyParser.json());

// Simple audit logger
async function audit(event) {
  const line = `[${new Date().toISOString()}] ${JSON.stringify(event)}\n`;
  await fs.appendFile("./mcp-audit.log", line);
}

// Middleware: authorize bearer
app.use((req, res, next) => {
  const auth = (req.headers.authorization || "").split(" ")[1];
  if (!auth || auth !== BEARER) {
    audit({ type: "unauth", ip: req.ip, path: req.path, headers: { ua: req.headers["user-agent"] } }).catch(()=>{});
    return res.status(401).json({ error: "Unauthorized" });
  }
  next();
});

/**
 * Utility: run a whitelisted shell command
 * - Commands are executed from repo root.
 * - Only whitelisted commands allowed to avoid arbitrary exec.
 */
const COMMAND_WHITELIST = {
  // key: name -> command executed on server (safe scripts)
  build: "pnpm build",
  test: "pnpm test --report",
  lint: "pnpm lint",
  typecheck: "pnpm tsc --noEmit",
  dev: "pnpm dev",
  "seed:db": "node scripts/seed.js"
};

app.post("/run", async (req, res) => {
  const { cmdKey, args = [], timeout = 60_000 } = req.body || {};
  const requester = req.headers["x-requester"] || "unknown";

  if (!cmdKey || !(cmdKey in COMMAND_WHITELIST)) {
    await audit({ type: "run_rejected", cmdKey, requester, ip: req.ip });
    return res.status(400).json({ error: "Invalid or non-whitelisted command key" });
  }

  const fullCmd = `${COMMAND_WHITELIST[cmdKey]} ${args.map(a => `"${String(a).replace(/"/g, '\\"')}"`).join(" ")}`.trim();

  // Additional safety: destructive commands must include x-confirm header set to "yes"
  const requiresConfirm = ["seed:db"].includes(cmdKey);
  if (requiresConfirm && req.headers["x-confirm"] !== "yes") {
    await audit({ type: "run_needs_confirm", cmdKey, requester, fullCmd, ip: req.ip });
    return res.status(409).json({ error: "Command requires explicit confirmation header x-confirm: yes" });
  }

  await audit({ type: "run_start", cmdKey, requester, fullCmd, ip: req.ip });

  exec(fullCmd, { maxBuffer: 10 * 1024 * 1024 }, async (err, stdout, stderr) => {
    await audit({
      type: "run_finish",
      cmdKey,
      requester,
      fullCmd,
      exitCode: err ? (err.code || "err") : 0,
      stdout: String(stdout).slice(0, 10000),
      stderr: String(stderr).slice(0, 10000),
    }).catch(()=>{});

    res.json({
      cmdKey,
      fullCmd,
      success: !err,
      stdout: String(stdout),
      stderr: String(stderr),
      error: err ? err.message : undefined
    });
  });
});

/** Tail logs (read last N lines) */
app.get("/logs/tail", async (req, res) => {
  const lines = parseInt(req.query.lines || "200", 10);
  const logfile = process.env.APP_LOG || "logs/app.log"; // default app log location
  try {
    const content = await fs.readFile(logfile, "utf8");
    const all = content.split(/\r?\n/).filter(Boolean);
    const tail = all.slice(-lines).join("\n");
    await audit({ type: "logs_tail", lines, requester: req.headers["x-requester"] || "unknown" });
    res.type("text/plain").send(tail);
  } catch (e) {
    await audit({ type: "logs_error", error: e.message, path: logfile }).catch(()=>{});
    res.status(500).json({ error: "Unable to read log file", details: e.message });
  }
});

/** Read file (safe root) */
app.post("/files/read", async (req, res) => {
  const { path: relPath } = req.body || {};
  if (!relPath) return res.status(400).json({ error: "path required" });

  // prevent path traversal; force inside project
  const safePath = path.resolve(process.cwd(), relPath);
  if (!safePath.startsWith(process.cwd())) return res.status(403).json({ error: "Forbidden" });

  try {
    const content = await fs.readFile(safePath, "utf8");
    await audit({ type: "files_read", path: relPath, requester: req.headers["x-requester"] || "unknown" });
    res.json({ path: relPath, content });
  } catch (e) {
    await audit({ type: "files_read_error", path: relPath, error: e.message }).catch(()=>{});
    res.status(500).json({ error: e.message });
  }
});

/** Write file (create/overwrite) -- requires x-confirm yes header */
app.post("/files/write", async (req, res) => {
  const { path: relPath, content } = req.body || {};
  if (!relPath) return res.status(400).json({ error: "path required" });

  const requiresConfirm = true; // force confirm for writes
  if (requiresConfirm && req.headers["x-confirm"] !== "yes") {
    await audit({ type: "files_write_needs_confirm", path: relPath, requester: req.headers["x-requester"] || "unknown" });
    return res.status(409).json({ error: "File writes require x-confirm: yes header" });
  }

  const safePath = path.resolve(process.cwd(), relPath);
  if (!safePath.startsWith(process.cwd())) return res.status(403).json({ error: "Forbidden" });

  try {
    await fs.mkdir(path.dirname(safePath), { recursive: true });
    await fs.writeFile(safePath, String(content || ""), "utf8");
    await audit({ type: "files_write", path: relPath, requester: req.headers["x-requester"] || "unknown" });
    res.json({ path: relPath, success: true });
  } catch (e) {
    await audit({ type: "files_write_error", path: relPath, error: e.message }).catch(()=>{});
    res.status(500).json({ error: e.message });
  }
});

/** Env get/set (wraps process.env) - writes require confirm and will be stored only in runtime (Replit Secrets should be used for persistent) */
app.post("/env/get", async (req, res) => {
  const { key } = req.body || {};
  if (!key) return res.status(400).json({ error: "key required" });
  await audit({ type: "env_get", key, requester: req.headers["x-requester"] || "unknown" });
  res.json({ key, value: process.env[key] || null });
});

app.post("/env/set", async (req, res) => {
  const { key, value } = req.body || {};
  if (!key) return res.status(400).json({ error: "key required" });
  if (req.headers["x-confirm"] !== "yes") {
    await audit({ type: "env_set_needs_confirm", key, requester: req.headers["x-requester"] || "unknown" });
    return res.status(409).json({ error: "Setting env requires x-confirm: yes" });
  }
  process.env[key] = String(value || "");
  await audit({ type: "env_set", key, requester: req.headers["x-requester"] || "unknown" });
  res.json({ success: true, key });
});

/** Create a GitHub PR from a branch with a patch (simple approach) */
app.post("/repo/pr", async (req, res) => {
  if (!GITHUB_TOKEN || !REPO_OWNER || !REPO_NAME) {
    return res.status(500).json({ error: "GitHub integration not configured (GITHUB_TOKEN/REPO_OWNER/REPO_NAME)" });
  }

  const { branchName, title, body } = req.body || {};
  if (!branchName || !title) return res.status(400).json({ error: "branchName and title required" });

  // This assumes the branch exists in the repo. For more complex flows, we'd push via git.
  const api = `https://api.github.com/repos/${REPO_OWNER}/${REPO_NAME}/pulls`;
  try {
    const r = await fetch(api, {
      method: "POST",
      headers: { "Authorization": `token ${GITHUB_TOKEN}`, "Accept": "application/vnd.github.v3+json" },
      body: JSON.stringify({ title, head: branchName, base: "main", body: body || "" })
    });
    const data = await r.json();
    await audit({ type: "repo_pr", branchName, title, result: { status: r.status, id: data.number } }).catch(()=>{});
    res.json({ ok: r.ok, status: r.status, data });
  } catch (e) {
    await audit({ type: "repo_pr_error", error: e.message }).catch(()=>{});
    res.status(500).json({ error: e.message });
  }
});

/** Simple ping */
app.get("/ping", async (req, res) => {
  await audit({ type: "ping", requester: req.headers["x-requester"] || "unknown" }).catch(()=>{});
  res.json({ ok: true, ts: Date.now() });
});

app.listen(PORT, () => {
  console.log(`MCP server listening on ${PORT}`);
});
