// IBrandBiz – Outline & Sections (Core Build)
// Minimal, production-ready scaffold to ship the Outline/Sections module
// Stack: React + TypeScript + Zustand + dnd-kit + shadcn/ui (buttons, inputs)
// This single file includes multiple virtual files, separated by markers.

/* =========================
   File: src/types/plan.ts
   ========================= */
export type SectionId = string;

export type SectionKind =
  | "executive-summary"
  | "company-overview"
  | "products-services"
  | "market-analysis"
  | "marketing-sales"
  | "operations"
  | "org-management"
  | "financials"
  | "custom";

export interface PlanSection {
  id: SectionId;
  kind: SectionKind;
  title: string;
  content: string; // markdown
  children?: PlanSection[]; // subsections (outline tree)
}

export interface TemplateField {
  id: string;
  label: string;
  type: "text" | "textarea" | "number" | "select";
  placeholder?: string;
  options?: string[]; // for select
}

export interface StructuredTemplate {
  key: "swot" | "persona" | "porter5";
  name: string;
  description: string;
  fields: TemplateField[];
}

export interface PlanState {
  planId: string;
  title: string;
  sections: PlanSection[];
}

/* =========================
   File: src/state/usePlanStore.ts
   ========================= */
import { create } from "zustand";
import { nanoid } from "nanoid";
import type { PlanState, PlanSection, SectionKind, SectionId } from "../types/plan";

const DEFAULT_SECTIONS: PlanSection[] = [
  { id: nanoid(), kind: "executive-summary", title: "Executive Summary", content: "" },
  { id: nanoid(), kind: "company-overview", title: "Company Overview", content: "" },
  { id: nanoid(), kind: "products-services", title: "Products & Services", content: "" },
  { id: nanoid(), kind: "market-analysis", title: "Market Analysis", content: "" },
  { id: nanoid(), kind: "marketing-sales", title: "Marketing & Sales", content: "" },
  { id: nanoid(), kind: "operations", title: "Operations Plan", content: "" },
  { id: nanoid(), kind: "org-management", title: "Organization & Management", content: "" },
  { id: nanoid(), kind: "financials", title: "Financial Plan", content: "" },
];

type Store = PlanState & {
  addSection(kind?: SectionKind, title?: string): SectionId;
  removeSection(id: SectionId): void;
  reorderSections(idsInOrder: SectionId[]): void;
  updateSection(id: SectionId, patch: Partial<PlanSection>): void;
  duplicateSection(id: SectionId): SectionId | null;
  resetToDefault(): void;
};

export const usePlanStore = create<Store>((set, get) => ({
  planId: nanoid(),
  title: "Untitled Business Plan",
  sections: DEFAULT_SECTIONS,

  addSection: (kind = "custom", title = "New Section") => {
    const section: PlanSection = { id: nanoid(), kind, title, content: "" };
    set((s) => ({ sections: [...s.sections, section] }));
    return section.id;
  },
  removeSection: (id) => set((s) => ({ sections: s.sections.filter((sec) => sec.id !== id) })),
  reorderSections: (ids) =>
    set((s) => ({
      sections: ids
        .map((id) => s.sections.find((x) => x.id === id)!)
        .filter(Boolean),
    })),
  updateSection: (id, patch) =>
    set((s) => ({ sections: s.sections.map((sec) => (sec.id === id ? { ...sec, ...patch } : sec)) })),
  duplicateSection: (id) => {
    const src = get().sections.find((x) => x.id === id);
    if (!src) return null;
    const dupe: PlanSection = { ...src, id: nanoid(), title: `${src.title} (Copy)` };
    set((s) => ({ sections: [...s.sections, dupe] }));
    return dupe.id;
  },
  resetToDefault: () => set({ sections: DEFAULT_SECTIONS }),
}));

/* =========================
   File: src/templates/structured.ts
   ========================= */
import type { StructuredTemplate } from "../types/plan";

export const SWOT_TEMPLATE: StructuredTemplate = {
  key: "swot",
  name: "SWOT Analysis",
  description: "Strengths, Weaknesses, Opportunities, Threats.",
  fields: [
    { id: "strengths", label: "Strengths", type: "textarea", placeholder: "List strengths..." },
    { id: "weaknesses", label: "Weaknesses", type: "textarea", placeholder: "List weaknesses..." },
    { id: "opportunities", label: "Opportunities", type: "textarea", placeholder: "List opportunities..." },
    { id: "threats", label: "Threats", type: "textarea", placeholder: "List threats..." },
  ],
};

export const PERSONA_TEMPLATE: StructuredTemplate = {
  key: "persona",
  name: "Customer Persona",
  description: "Capture ICP/persona details.",
  fields: [
    { id: "name", label: "Persona Name", type: "text", placeholder: "e.g., Side-Hustle Sarah" },
    { id: "demographics", label: "Demographics", type: "textarea" },
    { id: "jobs", label: "Jobs-To-Be-Done / Goals", type: "textarea" },
    { id: "pain", label: "Pain Points", type: "textarea" },
    { id: "channels", label: "Acquisition Channels", type: "textarea" },
  ],
};

export const STRUCTURED_TEMPLATES = [SWOT_TEMPLATE, PERSONA_TEMPLATE];

export function renderTemplateToMarkdown(key: string, data: Record<string, string>): string {
  if (key === "swot") {
    return `### SWOT Analysis\n\n**Strengths**\n${data.strengths || "-"}\n\n**Weaknesses**\n${data.weaknesses || "-"}\n\n**Opportunities**\n${data.opportunities || "-"}\n\n**Threats**\n${data.threats || "-"}`;
  }
  if (key === "persona") {
    return `### Customer Persona: ${data.name || "Unnamed"}\n\n**Demographics**\n${data.demographics || "-"}\n\n**Goals / JTBD**\n${data.jobs || "-"}\n\n**Pain Points**\n${data.pain || "-"}\n\n**Acquisition Channels**\n${data.channels || "-"}`;
  }
  return "";
}

/* =========================
   File: src/ai/prompts.ts
   ========================= */
export const OUTLINE_SUGGEST_PROMPT = (
  company: string,
  industry: string,
  audience: string
) => `You are an expert business analyst. Propose a concise outline for a business plan for a company named "${company}" in the ${industry} industry targeting ${audience}. 
Include 8 top-level sections (executive summary, company overview, products/services, market analysis, marketing & sales, operations, organization & management, financials). 
For each, return 3-6 bullet points with short labels only. Output JSON with {sections:[{title, bullets:[...] }]}.`;

export const SECTION_DRAFT_PROMPT = (
  sectionTitle: string,
  company: string,
  description: string,
  tone: "Professional" | "Friendly" | "Bold" | "Minimal"
) => `Write a ${tone.toLowerCase()} draft for the section "${sectionTitle}" of a business plan for ${company}. 
Keep it 150-250 words, clear and skimmable, using markdown paragraphs and short lists when helpful.`;

/* =========================
   File: src/components/OutlineEditor.tsx
   ========================= */
import React from "react";
import { DndContext, closestCenter, PointerSensor, useSensor, useSensors } from "@dnd-kit/core";
import { arrayMove, SortableContext, verticalListSortingStrategy } from "@dnd-kit/sortable";
import { SortableItem } from "./SortableItem";
import { usePlanStore } from "../state/usePlanStore";

export function OutlineEditor() {
  const sections = usePlanStore((s) => s.sections);
  const reorder = usePlanStore((s) => s.reorderSections);

  const sensors = useSensors(useSensor(PointerSensor));

  function onDragEnd(event: any) {
    const { active, over } = event;
    if (!over || active.id === over.id) return;
    const ids = sections.map((s) => s.id);
    const oldIndex = ids.indexOf(active.id);
    const newIndex = ids.indexOf(over.id);
    const newOrder = arrayMove(ids, oldIndex, newIndex);
    reorder(newOrder);
  }

  return (
    <DndContext sensors={sensors} collisionDetection={closestCenter} onDragEnd={onDragEnd}>
      <SortableContext items={sections.map((s) => s.id)} strategy={verticalListSortingStrategy}>
        <ul className="space-y-2">
          {sections.map((s) => (
            <SortableItem key={s.id} id={s.id} title={s.title} />
          ))}
        </ul>
      </SortableContext>
    </DndContext>
  );
}

/* =========================
   File: src/components/SortableItem.tsx
   ========================= */
import React from "react";
import { useSortable } from "@dnd-kit/sortable";
import { CSS } from "@dnd-kit/utilities";
import { usePlanStore } from "../state/usePlanStore";

export function SortableItem({ id, title }: { id: string; title: string }) {
  const { attributes, listeners, setNodeRef, transform, transition, isDragging } = useSortable({ id });
  const update = usePlanStore((s) => s.updateSection);
  const remove = usePlanStore((s) => s.removeSection);
  const duplicate = usePlanStore((s) => s.duplicateSection);

  const style: React.CSSProperties = {
    transform: CSS.Transform.toString(transform),
    transition,
    opacity: isDragging ? 0.6 : 1,
  };

  return (
    <li ref={setNodeRef} style={style} className="flex items-center justify-between rounded-2xl border p-3">
      <div className="flex items-center gap-3">
        <button aria-label="Drag" {...attributes} {...listeners} className="cursor-grab">≡</button>
        <input
          className="border-none outline-none bg-transparent text-base"
          defaultValue={title}
          onBlur={(e) => update(id, { title: e.currentTarget.value })}
        />
      </div>
      <div className="flex gap-2">
        <button onClick={() => duplicate(id)} className="px-2 py-1 rounded-lg border">Duplicate</button>
        <button onClick={() => remove(id)} className="px-2 py-1 rounded-lg border text-red-600">Delete</button>
      </div>
    </li>
  );
}

/* =========================
   File: src/components/StructuredForm.tsx
   ========================= */
import React, { useState } from "react";
import { STRUCTURED_TEMPLATES, renderTemplateToMarkdown } from "../templates/structured";
import { usePlanStore } from "../state/usePlanStore";

export function StructuredForm({ targetSectionId }: { targetSectionId: string }) {
  const [selectedKey, setSelectedKey] = useState<string>("swot");
  const [data, setData] = useState<Record<string, string>>({});
  const update = usePlanStore((s) => s.updateSection);

  const tpl = STRUCTURED_TEMPLATES.find((t) => t.key === selectedKey)!;

  function insert() {
    const md = renderTemplateToMarkdown(tpl.key, data);
    update(targetSectionId, (prev => ({ content: `${prev?.content ? prev.content + "\n\n" : ""}${md}` })) as any);
  }

  return (
    <div className="space-y-3">
      <select value={selectedKey} onChange={(e) => setSelectedKey(e.target.value)} className="border rounded-xl p-2">
        {STRUCTURED_TEMPLATES.map((t) => (
          <option key={t.key} value={t.key}>{t.name}</option>
        ))}
      </select>

      <div className="grid gap-2">
        {tpl.fields.map((f) => (
          <label key={f.id} className="grid gap-1">
            <span className="text-sm text-gray-600">{f.label}</span>
            {f.type === "textarea" ? (
              <textarea
                rows={3}
                className="rounded-xl border p-2"
                placeholder={f.placeholder}
                onChange={(e) => setData((d) => ({ ...d, [f.id]: e.target.value }))}
              />
            ) : (
              <input
                className="rounded-xl border p-2"
                placeholder={f.placeholder}
                onChange={(e) => setData((d) => ({ ...d, [f.id]: e.target.value }))}
              />
            )}
          </label>
        ))}
      </div>

      <button onClick={insert} className="rounded-2xl border px-3 py-2">Insert into Section</button>
    </div>
  );
}

/* =========================
   File: src/App.tsx (integration sketch)
   ========================= */
import React from "react";
import { OutlineEditor } from "./components/OutlineEditor";
import { usePlanStore } from "./state/usePlanStore";
import { StructuredForm } from "./components/StructuredForm";

export default function App() {
  const sections = usePlanStore((s) => s.sections);
  const update = usePlanStore((s) => s.updateSection);

  const first = sections[0];

  return (
    <div className="grid md:grid-cols-2 gap-6 p-6">
      <div>
        <h2 className="text-xl font-semibold mb-3">Outline</h2>
        <OutlineEditor />
        <div className="mt-4 flex gap-2 text-sm text-gray-600">
          <span>Drag to reorder, click title to rename, duplicate/delete with actions.</span>
        </div>
      </div>

      <div>
        <h2 className="text-xl font-semibold mb-3">Structured Templates</h2>
        {first && <StructuredForm targetSectionId={first.id} />}

        {first && (
          <div className="mt-6">
            <h3 className="font-medium mb-2">Live Preview (first section content)</h3>
            <textarea
              className="w-full h-48 rounded-xl border p-3"
              value={first.content}
              onChange={(e) => update(first.id, { content: e.target.value })}
            />
          </div>
        )}
      </div>
    </div>
  );
}
