// src/services/export/brandKitPdf.ts
import jsPDF from "jspdf";
// import html2canvas from "html2canvas"; // optional: capture DOM mockups

export type BrandTokens = {
  name: string;
  industry?: string;
  styles?: string[];
  palette: {
    primary: string; secondary: string; accent: string; highlight: string;
    neutral: string; surface: string; textLight: string; textDark: string;
  };
  fontFamily: string;   // display name (e.g., "Inter")
  textSize: number;     // base size shown in specimens
  iconUrl?: string|null;// generated SVG/PNG data URL
  slogans?: string[];   // optional
};

type Draw = (doc: jsPDF, x: number, y: number) => number;

const mm = (n: number) => n * 2.83465; // px→pt helper if you need it

// --- Shared decorations (header/footer, ribbon, swatch, section label) ---
function header(doc: jsPDF, titleLeft?: string, subtitleRight?: string) {
  doc.setFillColor("#0b1220"); // dark ribbon like your guide background blocks
  doc.rect(0, 0, doc.internal.pageSize.getWidth(), 48, "F");

  doc.setFont("helvetica", "bold"); doc.setTextColor("#ffffff"); doc.setFontSize(16);
  if (titleLeft)  doc.text(titleLeft, 40, 30);
  if (subtitleRight) {
    doc.setFont("helvetica", "normal");
    doc.text(subtitleRight, doc.internal.pageSize.getWidth() - 40, 30, { align: "right" });
  }
  doc.setTextColor("#111111"); // reset
}

function footer(doc: jsPDF, pg: number) {
  const h = doc.internal.pageSize.getHeight();
  doc.setDrawColor("#e5e7eb");
  doc.line(40, h - 52, doc.internal.pageSize.getWidth() - 40, h - 52);
  doc.setFont("helvetica", "normal"); doc.setFontSize(10); doc.setTextColor("#6b7280");
  doc.text(String(pg), doc.internal.pageSize.getWidth() - 40, h - 30, { align: "right" });
}

function sectionLabel(doc: jsPDF, big: string, small?: string) {
  doc.setFont("helvetica", "bold"); doc.setFontSize(22); doc.setTextColor("#111827");
  doc.text(big, 40, 88);
  if (small) { doc.setFont("helvetica", "normal"); doc.setFontSize(12); doc.setTextColor("#6b7280"); doc.text(small, 40, 106); }
  doc.setTextColor("#111111");
}

function swatch(doc: jsPDF, x: number, y: number, hex: string, label: string) {
  doc.setFillColor(hex); doc.rect(x, y, 36, 24, "F");
  doc.setFont("helvetica", "normal"); doc.setFontSize(11); doc.setTextColor("#111827");
  doc.text(`${label}: ${hex.toUpperCase()}`, x + 44, y + 16);
}

// Renders the composed mark (icon + wordmark) into a temporary canvas and returns PNG dataURL
async function composeMarkDataUrl(tokens: BrandTokens, layout: "side-by-side"|"top-bottom" = "side-by-side"): Promise<string|null> {
  if (!tokens.name) return null;
  const canvas = document.createElement("canvas");
  const ctx = canvas.getContext("2d");
  if (!ctx) return null;
  canvas.width = 1200; canvas.height = 600;
  ctx.clearRect(0,0,canvas.width,canvas.height);

  const iconSize = 220;
  const textSize = tokens.textSize || 72;
  const pad = 36;

  // draw icon
  let iconDrawn = false;
  if (tokens.iconUrl) {
    const img = new Image(); img.crossOrigin = "anonymous";
    await new Promise<void>((res, rej) => { img.onload = () => res(); img.onerror = () => res(); img.src = tokens.iconUrl!; });
    const cx = canvas.width/2, cy = canvas.height/2;
    if (layout === "side-by-side") {
      ctx.drawImage(img, cx - (iconSize + pad/2), cy - iconSize/2, iconSize, iconSize);
    } else {
      ctx.drawImage(img, cx - iconSize/2, cy - iconSize - 24, iconSize, iconSize);
    }
    iconDrawn = true;
  }

  // draw text
  ctx.fillStyle = tokens.palette.textDark;
  ctx.font = `bold ${textSize}px ${tokens.fontFamily}, system-ui, sans-serif`;
  ctx.textBaseline = "alphabetic";
  ctx.textAlign = "left";

  const text = tokens.name;
  const measure = ctx.measureText(text);
  const cx = canvas.width/2, cy = canvas.height/2;

  if (layout === "side-by-side") {
    const tx = iconDrawn ? cx + pad/2 : cx - measure.width/2;
    const ty = cy + iconSize/2; // baseline aligned with icon bottom
    ctx.fillText(text, tx, ty);
  } else {
    ctx.textAlign = "center";
    ctx.fillText(text, cx, cy + 24);
  }

  return canvas.toDataURL("image/png");
}

// --- Page factories ---
async function coverPage(doc: jsPDF, tokens: BrandTokens) {
  header(doc, "Brand Guidelines", "Make Your First Impression Your Best!");
  // Large center block
  const mid = doc.internal.pageSize.getWidth()/2;
  const y0 = 220;
  const mark = await composeMarkDataUrl(tokens);
  if (mark) doc.addImage(mark, "PNG", mid-200, y0-80, 400, 240, undefined, "FAST");
  doc.setFont("helvetica", "bold"); doc.setFontSize(28); doc.setTextColor("#111827");
  doc.text(tokens.name || "Your Brand", mid, y0 + 210, { align: "center" });
  doc.setFont("helvetica", "normal"); doc.setFontSize(12); doc.setTextColor("#6b7280");
  const sub = tokens.industry ? `${tokens.industry}` : "Brand Kit";
  doc.text(sub, mid, y0 + 235, { align: "center" });
  footer(doc, 1);
}

function welcomePage(doc: jsPDF) {
  header(doc, "01. Welcome", "IBrandBiz");
  sectionLabel(doc, "Welcome");
  doc.setFont("helvetica", "normal"); doc.setFontSize(12); doc.setTextColor("#111827");
  // Copy mirrors your guide’s welcome tone/structure
  doc.text(
    "Welcome to your Branding Guidelines. This document keeps your brand consistent across all platforms.", 
    40, 140, { maxWidth: 515 }
  );
  doc.text(
    "From logo usage and color palette to typography and applications, each element reflects your values and mission.",
    40, 164, { maxWidth: 515 }
  );
  // “Make Your First Impression Your Best!” callout
  doc.setFont("helvetica", "bold"); doc.setTextColor("#0b1220");
  doc.text("Make Your First Impression Your Best!", 40, 210);
  footer(doc, 2);
}

function tocPage(doc: jsPDF) {
  header(doc, "Table of Contents", "");
  sectionLabel(doc, "Contents");
  doc.setFont("helvetica", "normal"); doc.setFontSize(12);
  const lines = [
    "01. Welcome",
    "02. The Brand",
    "03. Palette",
    "04. Typography",
    "05. Logo & Icon",
    "06. Versions & Usage",
    "07. Mockups",
    "08. Assets",
    "09. Contact / Legal",
  ];
  lines.forEach((l, i) => doc.text(l, 40, 150 + i*20));
  footer(doc, 3);
}

function theBrandPage(doc: jsPDF, tokens: BrandTokens) {
  header(doc, "02. The Brand", "");
  sectionLabel(doc, "The Brand", "Vision • Values • Promise");
  doc.setFont("helvetica", "normal"); doc.setFontSize(12); doc.setTextColor("#111827");
  doc.text(`Brand: ${tokens.name}`, 40, 150);
  if (tokens.styles?.length) doc.text(`Style DNA: ${tokens.styles.join(", ")}`, 40, 170, { maxWidth: 515 });
  if (tokens.slogans?.length) {
    doc.setFont("helvetica","bold"); doc.text("Slogans", 40, 206);
    doc.setFont("helvetica","normal");
    tokens.slogans.slice(0,3).forEach((s, i) => doc.text(`• ${s}`, 40, 226 + i*16, { maxWidth: 515 }));
  }
  footer(doc, 4);
}

function paletteOverviewPage(doc: jsPDF, tokens: BrandTokens) {
  header(doc, "03. Palette", "");
  sectionLabel(doc, "Color Palette");
  const { palette } = tokens;
  const startY = 140, x = 40, col = 280;
  swatch(doc, x, startY,            palette.primary,   "Primary");
  swatch(doc, x, startY + 36,       palette.secondary, "Secondary");
  swatch(doc, x, startY + 72,       palette.accent,    "Accent");
  swatch(doc, x, startY + 108,      palette.highlight, "Highlight");
  swatch(doc, col, startY,          palette.neutral,   "Neutral");
  swatch(doc, col, startY + 36,     palette.surface,   "Surface");
  swatch(doc, col, startY + 72,     palette.textLight, "Text (Light)");
  swatch(doc, col, startY + 108,    palette.textDark,  "Text (Dark)");
  footer(doc, 5);
}

function typographyPage(doc: jsPDF, tokens: BrandTokens) {
  header(doc, "04. Typography", "");
  sectionLabel(doc, "Typography");
  doc.setFont("helvetica","normal"); doc.setFontSize(12);
  doc.text("Primary typeface & specimens:", 40, 140);
  // Large specimen
  doc.setFont(tokens.fontFamily, "bold"); // jsPDF uses built-ins unless you embed; see note below
  doc.setFontSize(28); doc.setTextColor("#111827");
  doc.text("Aa Bb Cc 123", 40, 176);
  doc.setFont("helvetica","normal"); doc.setFontSize(12); doc.setTextColor("#6b7280");
  doc.text(`Font: ${tokens.fontFamily}`, 40, 196);
  footer(doc, 6);
}

async function logoIconPage(doc: jsPDF, tokens: BrandTokens) {
  header(doc, "05. Logo & Icon", "");
  sectionLabel(doc, "Logo & Icon");
  const mark = await composeMarkDataUrl(tokens);
  if (mark) doc.addImage(mark, "PNG", 40, 150, 400, 220, undefined, "FAST");
  doc.setFont("helvetica","normal"); doc.setFontSize(11); doc.setTextColor("#6b7280");
  doc.text("Use clear space and sizing for readability. Provide RGB/CMYK assets for screen and print.", 40, 390, { maxWidth: 515 });
  footer(doc, 7);
}

function versionsUsagePage(doc: jsPDF) {
  header(doc, "06. Versions & Usage", "");
  sectionLabel(doc, "Versions & Usage");
  doc.setFont("helvetica","normal"); doc.setFontSize(12); doc.setTextColor("#111827");
  doc.text("Provide primary, reverse, mono, and icon variants. Keep contrast and minimum sizes in mind.", 40, 150, { maxWidth: 515 });
  doc.text("Export sizes: 72/96/150/300 ppi. Color profiles: RGB (screen) and CMYK (print).", 40, 174, { maxWidth: 515 });
  footer(doc, 8);
}

async function mockupsPage(doc: jsPDF, nodeId?: string) {
  header(doc, "07. Mockups", "");
  sectionLabel(doc, "Applications", "Business card • Website header • App icon");
  // If you have a DOM section with previews, capture it:
  // if (nodeId) {
  //   const n = document.getElementById(nodeId);
  //   if (n) {
  //     const canvas = await html2canvas(n, { backgroundColor: "#ffffff", scale: 2 });
  //     const img = canvas.toDataURL("image/png");
  //     doc.addImage(img, "PNG", 40, 160, 515, 320, undefined, "FAST");
  //   }
  // }
  footer(doc, 9);
}

function assetsPage(doc: jsPDF, tokens: BrandTokens) {
  header(doc, "08. Assets", "");
  sectionLabel(doc, "Downloadables");
  doc.setFont("helvetica","normal"); doc.setFontSize(12);
  doc.text("Included with this kit:", 40, 150);
  const items = [
    "SVG icon", "PNG logo (transparent)", "Composed logo PNG", "Color palette text file",
    "brand-info.json (tokens)", "Typeface guidance"
  ];
  items.forEach((t, i) => doc.text(`• ${t}`, 40, 172 + i*18));
  footer(doc, 10);
}

function contactLegalPage(doc: jsPDF) {
  header(doc, "09. Contact / Legal", "");
  sectionLabel(doc, "Contact & Legal");
  doc.setFont("helvetica","normal"); doc.setFontSize(12);
  doc.text("Questions or custom needs? Contact your IBrandBiz specialist.", 40, 150);
  doc.text("© IBrandBiz. All rights reserved.", 40, 176);
  footer(doc, 11);
}

// Public API
export async function createBrandKitPDF(tokens: BrandTokens, opts?: { mockupsNodeId?: string }) {
  const doc = new jsPDF({ unit: "pt", format: "a4" });

  await coverPage(doc, tokens);
  doc.addPage(); welcomePage(doc);
  doc.addPage(); tocPage(doc);
  doc.addPage(); theBrandPage(doc, tokens);
  doc.addPage(); paletteOverviewPage(doc, tokens);
  doc.addPage(); typographyPage(doc, tokens);
  doc.addPage(); await logoIconPage(doc, tokens);
  doc.addPage(); versionsUsagePage(doc);
  doc.addPage(); await mockupsPage(doc, opts?.mockupsNodeId); // optional screenshot
  doc.addPage(); assetsPage(doc, tokens);
  doc.addPage(); contactLegalPage(doc);

  // Page 12: blank notes or extra specimens
  doc.addPage();
  header(doc, "Notes", "");
  footer(doc, 12);

  const filename = `${(tokens.name||"brand").replace(/[^a-zA-Z0-9]/g,"-").toLowerCase()}-brand-kit.pdf`;
  doc.save(filename);
}
