import { useState, useEffect } from 'react';
import { DashboardTemplatePage } from '@/components/DashboardTemplatePage';
import { TemplateCard } from '@/components/templates/TemplateCard';
import { loadFirebaseTemplates, FirebaseLogoTemplate } from '@/services/firebase-templates';
import { Loader2, Search } from 'lucide-react';
import { Input } from '@/components/ui/input';

export default function LogoTemplates() {
  const [firebaseTemplates, setFirebaseTemplates] = useState<FirebaseLogoTemplate[]>([]);
  const [loading, setLoading] = useState(true);
  const [searchTerm, setSearchTerm] = useState('');

  // Load Firebase templates
  useEffect(() => {
    const loadTemplates = async () => {
      try {
        setLoading(true);
        const templates = await loadFirebaseTemplates();
        setFirebaseTemplates(templates);
      } catch (error) {
        console.error("Failed to load Firebase templates:", error);
      } finally {
        setLoading(false);
      }
    };

    loadTemplates();
  }, []);

  // Convert Firebase templates to display format
  const allTemplates = firebaseTemplates.map(template => ({
    id: template.id,
    name: template.name,
    description: template.description,
    tags: template.tags,
    previewUrl: template.assets?.previewPngPath ? `/firebase-storage/${template.assets.previewPngPath}` : undefined,
    source: 'firebase' as const
  }));

  // Filter templates based on search
  const filteredTemplates = allTemplates.filter(template => 
    template.name.toLowerCase().includes(searchTerm.toLowerCase()) ||
    template.description?.toLowerCase().includes(searchTerm.toLowerCase()) ||
    template.tags?.some(tag => tag.toLowerCase().includes(searchTerm.toLowerCase()))
  );

  const isLoading = loading;

  return (
    <DashboardTemplatePage title="Logo Templates">
      <div className="space-y-6">
        {/* Header Section */}
        <div className="bg-card p-6 rounded-lg border">
          <h2 className="text-xl font-semibold mb-2">Professional Logo Templates</h2>
          <p className="text-muted-foreground mb-4">
            Choose from our curated collection of professional logo templates. Customize colors, text, and positioning to match your brand.
          </p>
          
          {/* Search */}
          <div className="relative max-w-md">
            <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-muted-foreground w-4 h-4" />
            <Input
              placeholder="Search templates..."
              value={searchTerm}
              onChange={(e) => setSearchTerm(e.target.value)}
              className="pl-9"
            />
          </div>
        </div>

        {/* Templates Grid */}
        {isLoading ? (
          <div className="flex items-center justify-center py-12">
            <div className="text-center">
              <Loader2 className="w-8 h-8 animate-spin mx-auto mb-4 text-muted-foreground" />
              <p className="text-muted-foreground">Loading templates...</p>
            </div>
          </div>
        ) : filteredTemplates.length === 0 ? (
          <div className="text-center py-12">
            <p className="text-muted-foreground mb-2">
              {searchTerm ? `No templates found for "${searchTerm}"` : 'No templates available yet'}
            </p>
            {searchTerm ? (
              <button 
                onClick={() => setSearchTerm('')}
                className="text-sm text-blue-600 hover:text-blue-700"
              >
                Clear search
              </button>
            ) : (
              <p className="text-sm text-gray-500">
                Upload some logo templates from the admin panel to get started!
              </p>
            )}
          </div>
        ) : (
          <>
            {/* Template Count */}
            <div className="flex items-center justify-between">
              <p className="text-sm text-muted-foreground">
                {filteredTemplates.length} template{filteredTemplates.length !== 1 ? 's' : ''} available
              </p>
            </div>

            {/* Templates Grid */}
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-6">
              {filteredTemplates.map((template) => (
                <TemplateCard
                  key={`${template.source}-${template.id}`}
                  id={template.id}
                  name={template.name}
                  description={template.description}
                  tags={template.tags}
                  previewUrl={template.previewUrl}
                  source={template.source}
                />
              ))}
            </div>
          </>
        )}
      </div>
    </DashboardTemplatePage>
  );
}