import { useEffect, useMemo, useState } from "react";
import { useMutation } from "@tanstack/react-query";
import { apiRequest } from "@/lib/queryClient";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Skeleton } from "@/components/ui/skeleton";
import { Alert, AlertDescription } from "@/components/ui/alert";
import { Search, ExternalLink, CheckCircle, XCircle, AlertTriangle, Crown } from "lucide-react";
import { useToast } from "@/hooks/use-toast";

type Hit = {
  domain: string;
  available: boolean;
  priceCents?: number;       // integer cents (e.g., 1299 = $12.99)
  currency?: string;         // e.g., "USD"
  period?: number;           // years (default 1)
  premium?: boolean;         // premium domain flag
  definitive?: boolean;      // if provider not 100% certain
};

type SearchResponse = { hits: Hit[] };

const ENABLE_INTERNAL_CHECKOUT =
  (import.meta as any).env?.VITE_ENABLE_INTERNAL_DOMAIN_CHECKOUT === "true";

export default function FindDomainPage() {
  const [query, setQuery] = useState("");
  const [results, setResults] = useState<Hit[]>([]);
  const { toast } = useToast();

  // --- Search mutation (IBrandBiz backend) ---
  const searchMutation = useMutation({
    mutationFn: async (q: string) => {
      // backend contract: POST /api/domains/search { query: string }
      return await apiRequest<SearchResponse>("POST", "/api/domains/search", { query: q });
    },
    onSuccess: (data) => {
      setResults(data.hits || []);
    },
    onError: (err: any) => {
      toast({
        variant: "destructive",
        title: "Search failed",
        description: err?.message || "Unable to search domains right now.",
      });
    },
  });

  // ----- Prefill from URL: ?query= -----
  useEffect(() => {
    const params = new URLSearchParams(window.location.search);
    const q = params.get("query");
    if (q) {
      setQuery(q);
      // Clean URL
      window.history.replaceState({}, "", window.location.pathname);
      doSearch(q);
    }
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, []);

  // Normalizes user input (strip protocol/www, ensure TLD if desired)
  const normalize = (raw: string) => {
    let s = raw.trim().toLowerCase();
    s = s.replace(/^https?:\/\//, "").replace(/^www\./, "");
    s = s.split("/")[0];
    if (!s.includes(".")) s += ".com"; // default
    return s;
  };

  const doSearch = (raw?: string) => {
    const q = (raw ?? query).trim();
    if (!q) {
      toast({ variant: "destructive", title: "Enter a domain", description: "Try something like cafemojo.com" });
      return;
    }
    searchMutation.mutate(normalize(q));
  };

  const formatPrice = (cents?: number, currency: string = "USD") => {
    if (typeof cents !== "number") return "";
    return new Intl.NumberFormat("en-US", { style: "currency", currency }).format(cents / 100);
  };

  const statusFor = (hit: Hit) => {
    if (hit.definitive === false) return { text: "Uncertain", variant: "secondary" as const };
    return hit.available ? { text: "Available", variant: "default" as const } : { text: "Taken", variant: "destructive" as const };
  };

  // --- Buy / Checkout ---
  const buyDomain = async (domain: string) => {
    if (!ENABLE_INTERNAL_CHECKOUT) {
      // fallback: external GoDaddy search page
      const url = `https://www.godaddy.com/domainsearch/find?domainToCheck=${encodeURIComponent(domain)}`;
      window.open(url, "_blank", "noopener,noreferrer");
      return;
    }

    try {
      // minimal order payload; your backend will collect full registrant details in the checkout step
      const { checkoutUrl } = await apiRequest<{ checkoutUrl: string; orderId: string }>(
        "POST",
        "/api/domains/order",
        { domain, years: 1, privacy: true }
      );
      window.location.href = checkoutUrl; // Stripe Checkout (or your internal checkout route)
    } catch (e: any) {
      toast({
        variant: "destructive",
        title: "Checkout failed",
        description: e?.message || "Could not start checkout for this domain.",
      });
    }
  };

  // derived: any available hits?
  const anyAvail = useMemo(() => results.some(r => r.available), [results]);

  return (
    <div className="max-w-4xl mx-auto p-6">
      <div className="mb-8">
        <h1 className="text-3xl font-bold mb-2">Find a Domain</h1>
        <p className="text-muted-foreground">
          Search availability and secure your perfect web address. Tip: you can paste a full URL or just a name — we’ll clean it up.
        </p>
      </div>

      {/* Search */}
      <Card className="mb-8">
        <CardHeader>
          <CardTitle>Search</CardTitle>
          <CardDescription>Enter a domain to check availability and pricing</CardDescription>
        </CardHeader>
        <CardContent>
          <div className="flex gap-4">
            <Input
              className="flex-1"
              placeholder="e.g., cafemojo.com or cafemojo"
              value={query}
              onChange={(e) => setQuery(e.target.value)}
              onKeyDown={(e) => e.key === "Enter" && doSearch()}
              data-testid="input-domain-search"
            />
            <Button
              onClick={() => doSearch()}
              disabled={searchMutation.isPending}
              className="bg-green-600 hover:bg-green-700 text-white font-bold"
              data-testid="button-search-domains"
            >
              {searchMutation.isPending ? (
                <>
                  <div className="h-4 w-4 animate-spin rounded-full border-2 border-current border-t-transparent mr-2" />
                  Searching...
                </>
              ) : (
                <>
                  <Search className="h-4 w-4 mr-2" />
                  Search
                </>
              )}
            </Button>
          </div>
        </CardContent>
      </Card>

      {/* Loading */}
      {searchMutation.isPending && (
        <div className="space-y-4">
          <h2 className="text-xl font-semibold">Searching domains...</h2>
          <div className="grid gap-4">
            {[1, 2, 3, 4].map((i) => (
              <Card key={i}>
                <CardContent className="p-6">
                  <div className="flex items-center justify-between">
                    <div className="space-y-2 flex-1">
                      <Skeleton className="h-6 w-48" />
                      <Skeleton className="h-4 w-32" />
                    </div>
                    <Skeleton className="h-10 w-28" />
                  </div>
                </CardContent>
              </Card>
            ))}
          </div>
        </div>
      )}

      {/* Results */}
      {results.length > 0 && !searchMutation.isPending && (
        <div className="space-y-6">
          <div className="flex items-center justify-between">
            <h2 className="text-2xl font-semibold">Results</h2>
            {anyAvail ? (
              <span className="text-sm text-muted-foreground">Tip: lock an available domain before someone else does.</span>
            ) : (
              <span className="text-sm text-muted-foreground">No standard matches available—try a shorter name or another TLD.</span>
            )}
          </div>

          <div className="grid gap-4">
            {results.map((hit, idx) => {
              const status = statusFor(hit);
              const icon = hit.definitive === false
                ? <AlertTriangle className="h-5 w-5 text-yellow-500" />
                : hit.available
                  ? <CheckCircle className="h-5 w-5 text-green-500" />
                  : <XCircle className="h-5 w-5 text-red-500" />;

              return (
                <Card key={`${hit.domain}-${idx}`} data-testid={`domain-result-${idx}`}>
                  <CardContent className="p-6">
                    <div className="flex items-center justify-between">
                      <div className="flex items-center space-x-4">
                        {icon}
                        <div>
                          <h3 className="text-base font-semibold" data-testid={`domain-name-${idx}`}>{hit.domain}</h3>
                          <div className="flex items-center gap-2 mt-1">
                            <Badge variant={status.variant} data-testid={`domain-status-${idx}`}>
                              {status.text}
                            </Badge>

                            {hit.premium && (
                              <Badge variant="outline" className="gap-1">
                                <Crown className="h-3 w-3" /> Premium
                              </Badge>
                            )}

                            {hit.available && typeof hit.priceCents === "number" && (
                              <span className="text-base font-bold text-green-600" data-testid={`domain-price-${idx}`}>
                                {formatPrice(hit.priceCents, hit.currency || "USD")}/{hit.period || 1} yr
                              </span>
                            )}
                          </div>
                        </div>
                      </div>

                      {hit.available && (
                        <Button
                          onClick={() => buyDomain(hit.domain)}
                          className="text-white ml-4"
                          style={{ backgroundColor: '#FF8800' }}
                          onMouseEnter={(e) => (e.currentTarget.style.backgroundColor = '#E67700')}
                          onMouseLeave={(e) => (e.currentTarget.style.backgroundColor = '#FF8800')}
                          data-testid={`button-buy-${idx}`}
                        >
                          <ExternalLink className="h-4 w-4 mr-2" />
                          {ENABLE_INTERNAL_CHECKOUT ? "Checkout" : "Buy Now"}
                        </Button>
                      )}
                    </div>

                    {hit.definitive === false && (
                      <Alert className="mt-4">
                        <AlertTriangle className="h-4 w-4" />
                        <AlertDescription>
                          Availability is uncertain. Proceed to checkout to confirm.
                        </AlertDescription>
                      </Alert>
                    )}
                  </CardContent>
                </Card>
              );
            })}
          </div>
        </div>
      )}

      {/* Empty */}
      {results.length === 0 && !searchMutation.isPending && (
        <Card>
          <CardContent className="p-12 text-center">
            <Search className="h-12 w-12 text-muted-foreground mx-auto mb-4" />
            <h3 className="text-xl font-semibold mb-2">Start your domain search</h3>
            <p className="text-muted-foreground">
              Enter a domain above to check availability and see pricing.
            </p>
          </CardContent>
        </Card>
      )}
    </div>
  );
}
