# WHMCS Integration Guide

This document explains exactly how to port the preview styles into your WHMCS `ibrandbiz_comparison` orderform template.

## 🎯 Integration Strategy

**DO NOT** copy the entire HTML structure. WHMCS uses Smarty templates with specific data structures. Instead:

1. Copy **only CSS rules** to WHMCS stylesheet
2. Add **class hooks** to WHMCS template files
3. Let WHMCS handle all data/logic

---

## 📋 Step 1: Copy CSS to WHMCS

### File Location
```
/portal/templates/orderforms/ibrandbiz_comparison/style.css
```

### What to Copy

From `css/styles.css`, copy these sections:

#### A. CSS Variables (top of file)
```css
:root {
  --ibb-green: #00CB51;
  --ibb-orange: #FF8B00;
  --ibb-dark-green: #30342d;
  --ibb-dark-gray: #231f20;
  --ibb-blue: #274b9b;
  --ibb-olive: #978752;
  --ibb-light: #6dc282;
  --ibb-white: #ffffff;
}
```

#### B. Product Card Styles
Copy all rules starting with:
- `.product.ibb-card { ... }`
- `.product.featured { ... }`
- `.badge-pill { ... }`
- `.product-header.ibb-head { ... }`
- `.product-title.ibb-title { ... }`
- `.product-body { ... }`
- `.product-pricing { ... }`
- `.price-wrapper { ... }`
- `.price { ... }`
- `.cycle { ... }`
- `.annual-price { ... }`
- `.product-description { ... }`
- `.product-features.ibb-feats { ... }`
- `.product-note { ... }`
- `.btn-choose { ... }`

#### C. Order Summary Styles
Copy all rules starting with:
- `.order-summary.ibb-summary { ... }`
- `.summary-header { ... }`
- `.summary-body { ... }`
- `.summary-items { ... }`
- `.cart-item { ... }`
- `.summary-total { ... }`
- `.total-row { ... }`
- `.total-price { ... }`
- `.btn-continue { ... }`

#### D. Domain Search Styles (if using in WHMCS)
- `.domain-search-bar { ... }`
- `.domain-input { ... }`
- `.btn-search { ... }`
- `.btn-transfer { ... }`

#### E. Responsive Rules
Copy the `@media` queries at the bottom

### What NOT to Copy
- ❌ Header/navbar styles (WHMCS client theme handles this)
- ❌ Footer styles (WHMCS client theme handles this)
- ❌ Body/global styles that conflict with WHMCS

---

## 🔧 Step 2: Add Class Hooks to WHMCS Templates

### Template File Locations
```
/portal/templates/orderforms/ibrandbiz_comparison/
├── products.tpl
├── viewcart.tpl
└── configureproduct.tpl
```

### A. products.tpl (Main Product Grid)

#### 1. Products Container
Find the products grid wrapper, add class:
```smarty
<div class="row ibb-products">
  {foreach $products as $product}
    ...
  {/foreach}
</div>
```

#### 2. Product Card Root
Find each product card, add classes:
```smarty
<div class="product ibb-card {if $product.isFeatured}featured{/if}">
  ...
</div>
```

#### 3. Featured Badge
For featured products, add badge before header:
```smarty
{if $product.isFeatured}
  <span class="badge-pill">★ Recommended</span>
{/if}
```

#### 4. Product Header
```smarty
<div class="product-header ibb-head">
  <h3 class="product-title ibb-title">{$product.name}</h3>
</div>
```

#### 5. Product Body
```smarty
<div class="product-body">
  <div class="product-pricing">
    <div class="price-wrapper">
      <span class="price">${$product.pricing.monthly}</span>
      <span class="cycle">/month</span>
    </div>
    {if $product.pricing.annually}
      <div class="annual-price">${$product.pricing.annually} /year</div>
    {/if}
  </div>
  
  <p class="product-description">{$product.tagline}</p>
  
  <ul class="product-features ibb-feats">
    {foreach $product.features as $feature}
      <li><i class="fas fa-check"></i> {$feature}</li>
    {/foreach}
  </ul>
  
  {if $product.description}
    <p class="product-note">{$product.description}</p>
  {/if}
  
  <button class="btn btn-primary btn-block btn-choose" data-product-id="{$product.id}">
    Choose Plan
  </button>
</div>
```

### B. viewcart.tpl (Cart/Checkout Page)

#### 1. Page Wrapper
Add class to main cart container:
```smarty
<div class="cart-page ibb-cart">
  ...
</div>
```

#### 2. Order Summary Panel
Find the order summary/totals section:
```smarty
<div class="order-summary ibb-summary">
  <div class="summary-header">
    <h3>Order Summary</h3>
  </div>
  <div class="summary-body">
    <div class="summary-items">
      {foreach $cartitems as $item}
        <div class="cart-item">
          <span class="cart-item-name">{$item.productname}</span>
          <span class="cart-item-price">${$item.amount}</span>
        </div>
      {/foreach}
    </div>
    <div class="summary-total">
      <div class="total-row">
        <span>Total:</span>
        <span class="total-price">${$total}</span>
      </div>
    </div>
  </div>
</div>
```

### C. configureproduct.tpl (Product Configuration)

Add wrapper class:
```smarty
<div class="config-page ibb-config">
  ...
</div>
```

---

## 🚫 What to Leave to WHMCS

**DO NOT hard-code** these values in templates:
- Product names (use `{$product.name}`)
- Prices (use `{$product.pricing.*}`)
- Billing cycles (use `{$product.billingcycle}`)
- Product descriptions (use `{$product.description}`)
- Add-to-cart URLs (use `{$product.orderurl}`)
- Feature lists (use `{$product.features}`)

WHMCS dynamically populates all data via Smarty variables.

---

## ⚙️ Step 3: WHMCS Configuration

### 1. Set Default Order Form
```
WHMCS Admin → System Settings → General Settings → Ordering
→ Default Order Form Template: ibrandbiz_comparison
```

### 2. Clear Template Cache
```
WHMCS Admin → Utilities → System → System Cleanup
→ Empty Template Cache
```

### 3. Test the Integration
```
Visit: https://billing.ibrandbiz.com/cart.php
```

**Check:**
- ✅ Product cards display correctly
- ✅ Featured product has ★ badge and green border
- ✅ Pricing shows properly formatted
- ✅ Features list with green checkmarks
- ✅ Order summary updates on selection
- ✅ Responsive on mobile/tablet

---

## 🎨 Asset Alignment

### Bootstrap Version
- Preview uses: Bootstrap 3.4.1
- WHMCS Six uses: Bootstrap 3.4.1
- ✅ **Perfect match** - no conflicts

### Font Awesome
- Preview uses: Font Awesome 5.x
- WHMCS Six ships: Font Awesome (check version)
- If older version, update icon classes:
  - `fas fa-check` → `fa fa-check`
  - `fas fa-arrow-right` → `fa fa-arrow-right`

### CSS Specificity
Our custom classes (`.ibb-*`) won't conflict with:
- WHMCS default `.panel`, `.navbar`, etc.
- Client theme styles
- Bootstrap core classes

---

## 🔍 Selector Mapping Reference

| Preview Class | WHMCS Usage | Template File |
|--------------|-------------|---------------|
| `.ibb-products` | Products grid wrapper | products.tpl |
| `.product.ibb-card` | Individual product card | products.tpl |
| `.product.featured` | Featured product (conditional) | products.tpl |
| `.badge-pill` | "★ Recommended" badge | products.tpl |
| `.product-header.ibb-head` | Card header section | products.tpl |
| `.product-title.ibb-title` | Product name/title | products.tpl |
| `.product-features.ibb-feats` | Features list | products.tpl |
| `.ibb-summary` | Order summary sidebar | viewcart.tpl |
| `.ibb-cart` | Cart page wrapper | viewcart.tpl |
| `.ibb-config` | Config page wrapper | configureproduct.tpl |

---

## ✅ Post-Integration Checklist

After porting to WHMCS:

- [ ] CSS rules copied to `style.css`
- [ ] Class hooks added to templates
- [ ] No hard-coded prices/names in templates
- [ ] Default order form set to `ibrandbiz_comparison`
- [ ] Template cache cleared
- [ ] Desktop view tested
- [ ] Mobile view tested
- [ ] Featured product displays ★ badge
- [ ] Order summary updates correctly
- [ ] Add-to-cart functionality works
- [ ] No JavaScript errors in console
- [ ] No CSS conflicts with client theme

---

## 🆘 Troubleshooting

### Issue: Featured badge not showing
**Fix:** Verify WHMCS has `isFeatured` flag set on product. Check with:
```smarty
{if $product.isFeatured}Featured!{/if}
```

### Issue: Prices not displaying
**Fix:** Check Smarty variable names. WHMCS might use:
- `{$product.pricing.monthly}` or
- `{$product.pricing.rawpricing.monthly}` or
- `{$product.monthlypricing}`

### Issue: Checkmarks not appearing
**Fix:** Ensure Font Awesome is loaded. Add to header if needed:
```html
<link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/5.15.4/css/all.min.css">
```

### Issue: Layout broken on mobile
**Fix:** Verify Bootstrap grid classes:
- Desktop: `col-md-8` (products) + `col-md-4` (summary)
- Mobile: Should stack automatically

---

## 📚 Additional Resources

- WHMCS Orderform Docs: https://docs.whmcs.com/Order_Form_Templates
- WHMCS Smarty Variables: https://docs.whmcs.com/Template_Variables
- Bootstrap 3 Grid: https://getbootstrap.com/docs/3.4/css/#grid

---

**Questions?** Refer back to the preview files for exact implementation details.
